# Including voice feature
import json
import os
import re
import sqlite3
import requests
import tempfile  # 🔊 for saving downloaded audio

import pandas as pd
from flask import Flask, render_template, request, jsonify, session
from nlp_utils import detect_intent, detect_language
from openai import OpenAI
from dotenv import load_dotenv
import whisper  # 🔊 Offline Whisper (FREE)

load_dotenv()

# ---------------------------
# ENV VARIABLES
# ---------------------------
OPENROUTER_API_KEY = os.getenv("OPENROUTER_API_KEY", "YOUR_OPENROUTER_API_KEY_HERE")
FLASK_SECRET = os.getenv("FLASK_SECRET_KEY", "joana-fastfood-secret")

# WhatsApp Cloud API envs
WHATSAPP_TOKEN = os.getenv("WHATSAPP_TOKEN")
WHATSAPP_PHONE_NUMBER_ID = os.getenv("WHATSAPP_PHONE_NUMBER_ID")
WHATSAPP_VERIFY_TOKEN = os.getenv("WHATSAPP_VERIFY_TOKEN", "joana-verify-token-123")

WHATSAPP_API_BASE = "https://graph.facebook.com/v21.0"

# 🔊 Load Whisper model once
try:
    whisper_model = whisper.load_model("small")  # 'tiny', 'base', 'small', 'medium', 'large'
    print("✅ Offline Whisper model loaded: small")
except Exception as e:
    whisper_model = None
    print("❌ Whisper load failed:", e)


# ---------------------------
# WHATSAPP CLOUD SEND HELPERS
# ---------------------------


def send_whatsapp_text(to_number: str, text: str):
    """
    Send plain text via WhatsApp Cloud API
    """
    print("=== send_whatsapp_text CALLED ===")
    print("WHATSAPP_TOKEN present?", bool(WHATSAPP_TOKEN))
    print("Sending to:", to_number, "| Text:", text)

    if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
        print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
        return

    url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
    headers = {
        "Authorization": f"Bearer {WHATSAPP_TOKEN}",
        "Content-Type": "application/json",
    }
    payload = {
        "messaging_product": "whatsapp",
        "to": to_number,
        "type": "text",
        "text": {"body": text},
    }

    try:
        resp = requests.post(url, headers=headers, json=payload, timeout=30)
        print("CLOUD SEND RESPONSE:", resp.status_code, resp.text)
    except Exception as e:
        print("Error sending WhatsApp text via Cloud API:", repr(e))


def send_whatsapp_image(to_number: str, image_url: str, caption: str = ""):
    """
    Send image + optional caption via WhatsApp Cloud API
    """
    if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
        print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
        return False

    url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
    headers = {
        "Authorization": f"Bearer {WHATSAPP_TOKEN}",
        "Content-Type": "application/json",
    }
    payload = {
        "messaging_product": "whatsapp",
        "to": to_number,
        "type": "image",
        "image": {
            "link": image_url,
        },
    }
    if caption:
        payload["image"]["caption"] = caption

    try:
        resp = requests.post(url, headers=headers, json=payload, timeout=30)
        print("CLOUD IMAGE SEND:", resp.status_code, resp.text)
        return resp.status_code in (200, 201)
    except Exception as e:
        print("Error sending WhatsApp image via Cloud API:", e)
        return False


def send_whatsapp_quick_buttons(to_number: str, body_text: str, buttons: list):
    """
    Send interactive "button" template via WhatsApp Cloud API
    (max 3 buttons)
    """
    if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
        print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
        return False

    url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
    headers = {
        "Authorization": f"Bearer {WHATSAPP_TOKEN}",
        "Content-Type": "application/json",
    }

    # Cloud API button format
    wa_buttons = []
    for b in buttons[:3]:
        wa_buttons.append(
            {
                "type": "reply",
                "reply": {
                    "id": b.get("id", b["title"]),
                    "title": b["title"],
                },
            }
        )

    payload = {
        "messaging_product": "whatsapp",
        "to": to_number,
        "type": "interactive",
        "interactive": {
            "type": "button",
            "body": {"text": body_text},
            "action": {"buttons": wa_buttons},
        },
    }

    try:
        resp = requests.post(url, headers=headers, json=payload, timeout=30)
        print("CLOUD BUTTONS SEND:", resp.status_code, resp.text)
        return resp.status_code in (200, 201)
    except Exception as e:
        print("Error sending WhatsApp buttons via Cloud API:", e)
        return False


# ---------------------------
# Paths / Directories
# ---------------------------
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
DATA_DIR = os.path.join(BASE_DIR, "data")

MENU_FILE = os.path.join(DATA_DIR, "Menu.xlsx")
BRANCHES_FILE = os.path.join(DATA_DIR, "Branches.xlsx")

WHATSAPP_SESSIONS = {}
WA_CATEGORY_STATE = {}  # phone -> {"category": str, "index": int}

# ---------------------------
# Flask Config
# ---------------------------
app = Flask(__name__, static_folder="static", template_folder="templates")
app.secret_key = FLASK_SECRET

client = OpenAI(api_key=OPENROUTER_API_KEY, base_url="https://openrouter.ai/api/v1")

PAYMENT_URL = (
    "https://starlit-sopapillas-520aa2.netlify.app/"
    "?redirect=http://127.0.0.1:5000/thankyou"
)
CURRENCY = "SAR"
CURRENCY_AR = "ريال سعودي"

# ---------------------------
# Database
# ---------------------------
DB_PATH = os.path.join(DATA_DIR, "orders.db")


def init_db():
    os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    conn.execute(
        """
        CREATE TABLE IF NOT EXISTS orders (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            item TEXT,
            qty INTEGER,
            spicy INTEGER,
            nonspicy INTEGER,
            price REAL,
            subtotal REAL,
            total REAL,
            payment_method TEXT,
            datetime TEXT
        )
        """
    )
    conn.commit()
    conn.close()


init_db()

# ---------------------------
# Category mapping helper
# ---------------------------


def canonical_category(raw_category: str, name_en: str = "") -> str:
    """
    Excel category ko 3 main groups me convert:
    burgers_meals, sandwiches, snacks_sides
    """
    c = (raw_category or "").strip().lower()
    n = (name_en or "").strip().lower()

    if c in ["burgers", "burger", "meals", "meal"]:
        return "burgers_meals"
    if c == "sandwiches":
        return "sandwiches"
    if c in ["sides", "side", "drinks", "drink"]:
        return "snacks_sides"

    if any(
        k in n for k in ["burger", "zinger", "tortilla", "hot dog", "hotdog", "meal"]
    ):
        return "burgers_meals"
    if any(k in n for k in ["sandwich", "kudu", "kabab"]):
        return "sandwiches"
    if any(
        k in n
        for k in [
            "fries",
            "nuggets",
            "rings",
            "juice",
            "corn",
            "potato",
            "samosa",
            "popcorn",
        ]
    ):
        return "snacks_sides"

    return ""


# ---------------------------
# Excel Loaders
# ---------------------------


def load_menu():
    try:
        df_raw = pd.read_excel(MENU_FILE, header=None)
        header_row_index = None

        for i, row in df_raw.iterrows():
            if any("name_en" in str(c).lower() for c in row):
                header_row_index = i
                break

        if header_row_index is None:
            print("❌ Could not find header row containing 'name_en'")
            return {}

        df = pd.read_excel(MENU_FILE, header=header_row_index)

        def has(col, *keys):
            c = str(col).strip().lower()
            return any(k in c for k in keys)

        english_col = next((c for c in df.columns if has(c, "name_en")), None)
        arabic_col = next((c for c in df.columns if has(c, "name_ar")), None)
        price_col = next((c for c in df.columns if has(c, "price")), None)
        cat_col = next(
            (c for c in df.columns if has(c, "category", "cat", "type")), None
        )

        if not english_col or not price_col:
            raise Exception("Missing name_en or price column")

        menu = {}
        for _, row in df.iterrows():
            en = str(row[english_col]).strip().lower()
            if not en or en == "nan":
                continue

            ar = str(row[arabic_col]).strip() if arabic_col else en
            raw_cat = str(row.get(cat_col, "")).strip() if cat_col else ""

            try:
                price = float(row[price_col])
            except Exception:
                continue

            cat_final = canonical_category(raw_cat, en)

            entry = {
                "price": price,
                "category": cat_final,
                "name_en": en,
                "name_ar": ar or en,
            }

            menu[en] = entry
            if ar:
                menu[ar.lower()] = entry

        print(f"✅ Loaded {len(menu)} menu items.")
        return menu

    except Exception as e:
        print("Menu load failed:", e)
        return {}


def load_branches():
    try:
        df_raw = pd.read_excel(BRANCHES_FILE, header=None)
        header_row_index = None

        for i, row in df_raw.iterrows():
            row_l = [str(c).lower() for c in row]
            if any("branch" in c for c in row_l) and any("address" in c for c in row_l):
                header_row_index = i
                break

        if header_row_index is None:
            header_row_index = 0

        df = pd.read_excel(BRANCHES_FILE, header=header_row_index)

        name_col = next((c for c in df.columns if "branch" in str(c).lower()), None)
        addr_col = next((c for c in df.columns if "address" in str(c).lower()), None)
        phone_col = next(
            (
                c
                for c in df.columns
                if "phone" in str(c).lower() or "number" in str(c).lower()
            ),
            None,
        )

        branches = []
        for _, row in df.iterrows():
            branches.append(
                {
                    "Branch Name": str(row.get(name_col, "")).strip(),
                    "Address / Area": str(row.get(addr_col, "")).strip(),
                    "Phone Number": str(row.get(phone_col, "")).strip(),
                }
            )

        print(f"✅ Loaded {len(branches)} branches.")
        return [
            b
            for b in branches
            if (b["Branch Name"] or b["Address / Area"] or b["Phone Number"])
        ]

    except Exception as e:
        print("❌ Branch load failed:", e)
        return []


MENU = load_menu()
BRANCHES = load_branches()

# ---------------------------
# Helpers
# ---------------------------


def build_menu_context():
    if not MENU:
        return "Current restaurant menu is empty."

    lines = []
    for name, info in MENU.items():
        if not re.search(r"[A-Za-z]", name):
            continue
        price = info.get("price", 0.0)
        category = info.get("category", "") or ""
        lines.append(f"- {name} | price: {price:.2f} {CURRENCY} | category: {category}")

    if not lines:
        return "Current restaurant menu is empty."

    return "Current restaurant menu (items, prices, categories):\n" + "\n".join(lines)


def build_branches_message(lang: str = "en") -> str:
    lines = []
    for b in BRANCHES[:6]:
        name = b.get("Branch Name", "")
        addr = b.get("Address / Area", "")
        phone = b.get("Phone Number", "")
        if phone:
            lines.append(f"- {name} – {addr} | {phone}")

    branches_text = "\n".join(lines) if lines else "Branches info not available."

    if lang == "ar":
        return (
            "مرحباً بك في مطعم جوانا للوجبات السريعة!\n\n"
            "الفروع:\n"
            f"{branches_text}\n\n"
            "من فضلك اختر طريقة الطلب:"
        )
    else:
        return (
            "Welcome to JOANA Fast Food! \n\n"
            "Branches:\n"
            f"{branches_text}\n\n"
            "Please choose an option:"
        )


def html_to_whatsapp(text: str) -> str:
    if not text:
        return ""

    text = text.replace("<br>", "\n").replace("<br/>", "\n").replace("<br />", "\n")

    def replace_anchor(match):
        full_tag = match.group(0)
        label = match.group(1).strip()

        href_match = re.search(
            r'href=[\'"]([^\'"]+)[\'"]', full_tag, flags=re.IGNORECASE
        )
        if not href_match:
            return label

        url = href_match.group(1).strip()

        if url.startswith("tel:"):
            return label

        return f"{label}: {url}"

    text = re.sub(
        r"<a[^>]>(.?)</a>", replace_anchor, text, flags=re.IGNORECASE | re.DOTALL
    )
    text = re.sub(r"<[^>]+>", "", text)
    return text.strip()


def normalize_number(raw: str) -> str:
    if not raw:
        return ""
    raw = str(raw)
    if raw.startswith("+"):
        raw = raw[1:]
    return raw


WA_GREETINGS = [
    "hi",
    "hii",  # NEW
    "hello",
    "hey",
    "salam",
    "slam",
    "asalam",
    "assalam",
    "assalam o alaikum",
    "assalamu alaikum",
    "السلام عليكم",
    "مرحبا",
]


def is_wa_greeting(text: str) -> bool:
    """
    WhatsApp greeting sirf tab jab pura text hi greeting ho.
    'chicken sandwich' mein 'hi' aata hai lekin yeh greeting nahi hai.
    """
    if not text:
        return False
    t = text.strip().lower()
    t = re.sub(r"[^\w\u0600-\u06FF ]+", "", t)
    return t in WA_GREETINGS


def llm_normalize_text(text: str, lang: str) -> str:
    if not text or lang != "ar":
        return text
    # If OpenRouter key invalid / missing, skip LLM normalization
    if not OPENROUTER_API_KEY or OPENROUTER_API_KEY.startswith("YOUR_"):
        return text
    try:
        res = client.chat.completions.create(
            model="gpt-4o-mini",
            messages=[
                {
                    "role": "system",
                    "content": (
                        "You are a spell-correction engine for user messages.\n"
                        "- Fix Arabic spelling, don't translate.\n"
                        "- Return only corrected text."
                    ),
                },
                {"role": "user", "content": text},
            ],
            temperature=0,
            max_tokens=100,
        )
        return res.choices[0].message.content.strip()
    except Exception as e:
        print("LLM normalize error:", e)
        return text


def detect_qty(msg: str) -> int:
    if not msg:
        return 1
    text = msg.lower().strip()

    arabic_digit_map = {
        "٠": "0",
        "١": "1",
        "٢": "2",
        "٣": "3",
        "٤": "4",
        "٥": "5",
        "٦": "6",
        "٧": "7",
        "٨": "8",
        "٩": "9",
    }
    for ar, en in arabic_digit_map.items():
        text = text.replace(ar, en)

    digit_match = re.search(r"\b(\d+)\b", text)
    if digit_match:
        try:
            q = int(digit_match.group(1))
            if q > 0:
                return q
        except ValueError:
            pass

    number_words_en = {
        "one": 1,
        "two": 2,
        "three": 3,
        "four": 4,
        "five": 5,
        "six": 6,
        "seven": 7,
        "eight": 8,
        "nine": 9,
        "ten": 10,
    }
    for word, value in number_words_en.items():
        if re.search(r"\b" + re.escape(word) + r"\b", text):
            return value

    number_words_ar = {
        "واحد": 1,
        "واحدة": 1,
        "اثنين": 2,
        "اتنين": 2,
        "ثنين": 2,
        "ثلاثة": 3,
        "ثلاث": 3,
        "اربعة": 4,
        "أربعة": 4,
        "خمسة": 5,
        "ستة": 6,
        "سبعة": 7,
        "ثمانية": 8,
        "تسعة": 9,
        "عشرة": 10,
    }
    for word, value in number_words_ar.items():
        if word in text:
            return value

    return 1


def polite_check(text):
    bad = ["idiot", "stupid", "حرام", "لعنة", "غبي"]
    return any(w in text.lower() for w in bad)


def get_price_and_category(name):
    entry = MENU.get(name.lower(), {})
    return float(entry.get("price", 0)), entry.get("category", "")


def build_order_summary_and_total(order_items, lang):
    total = sum(i.get("subtotal", 0) for i in order_items)
    summary = []

    for i in order_items:
        item_name = i["item"]
        item_info = MENU.get(item_name.lower(), {})
        category = item_info.get("category")

        if category == "burgers_meals":
            if lang == "ar":
                kind = "حار" if i.get("spicy") else "بدون حار"
            else:
                kind = "spicy" if i.get("spicy") else "non-spicy"
            summary.append(f"{i['qty']} {kind} {item_name.title()}")
        else:
            summary.append(f"{i['qty']} {item_name.title()}")

    return summary, total


def speak_text(reply, lang):
    return


# 🔊 Voice transcription helper for WhatsApp Cloud
def transcribe_audio_from_cloud(audio_obj: dict) -> str:
    """
    audio_obj me se media id nikal kar:
    1) Cloud API se media URL lete hain
    2) file download karte hain
    3) Whisper se text me convert karte hain
    """
    if not audio_obj:
        print("No audio_obj passed to transcribe_audio_from_cloud.")
        return ""

    media_id = audio_obj.get("id")
    if not media_id:
        print("No media_id in audio_obj.")
        return ""

    if not WHATSAPP_TOKEN:
        print("WHATSAPP_TOKEN missing, cannot download media.")
        return ""

    if not whisper_model:
        print("Whisper model not loaded, cannot transcribe.")
        return ""

    try:
        # Step 1: Get media URL from Cloud API
        meta_url = f"{WHATSAPP_API_BASE}/{media_id}"
        headers = {"Authorization": f"Bearer {WHATSAPP_TOKEN}"}
        meta_resp = requests.get(meta_url, headers=headers, timeout=30)
        print("Media meta status:", meta_resp.status_code, meta_resp.text)

        if meta_resp.status_code != 200:
            return ""

        media_url = meta_resp.json().get("url")
        if not media_url:
            print("No media URL returned from meta endpoint.")
            return ""

        # Step 2: Download media binary
        file_resp = requests.get(media_url, headers=headers, timeout=60)
        if file_resp.status_code != 200:
            print("Media download failed:", file_resp.status_code)
            return ""

        # Step 3: Save to temp file
        with tempfile.NamedTemporaryFile(suffix=".ogg", delete=False) as tmp:
            tmp.write(file_resp.content)
            tmp_path = tmp.name

        print("Transcribing audio file:", tmp_path)
        result = whisper_model.transcribe(tmp_path)
        text = result.get("text", "").strip()
        print("Whisper transcription:", text)
        return text

    except Exception as e:
        print("Error in transcribe_audio_from_cloud:", repr(e))
        return ""


def parse_spice_split(msg):
    text = msg.lower().replace("-", " ")
    nums = re.findall(r"\d+", text)

    if len(nums) >= 2:
        n1, n2 = int(nums[0]), int(nums[1])

        if ("spicy" in text or "حار" in text) and (
            "non" in text or "mild" in text or "عادي" in text or "بدون" in text
        ):
            spicy_first = (
                text.find("spicy") < text.find("non") if "non" in text else True
            )
            arab_spicy_first = (
                text.find("حار") < text.find("عادي") if "عادي" in text else True
            )

            return (n1, n2) if (spicy_first or arab_spicy_first) else (n2, n1)

    return None


def find_menu_item(msg):
    text = msg.lower()
    candidates = []

    for name in MENU.keys():
        pattern = r"\b" + re.escape(name) + r"s?\b"
        if re.search(pattern, text):
            candidates.append(name)

    if not candidates:
        return None

    candidates.sort(key=len, reverse=True)
    return candidates[0]


def get_items_by_category(category: str):
    """
    Return unique items for a category:
    [{"en": "...", "ar": "..."}]
    """
    items_map = {}
    for key, info in MENU.items():
        if info.get("category") != category:
            continue
        en = (info.get("name_en") or key).strip().lower()
        ar = (info.get("name_ar") or en).strip()
        items_map[en] = {"en": en, "ar": ar}
    return list(items_map.values())


def detect_spicy_nonspicy(msg: str):
    """
    Text se spicy / non-spicy flags nikalta hai.
    """
    if not msg:
        return False, False

    text = msg.lower()

    nonspicy_keywords_en = [
        "non spicy",
        "non-spicy",
        "no spicy",
        "without spicy",
        "without spice",
        "not spicy",
        "mild",
    ]
    nonspicy_keywords_ar = [
        "بدون حار",
        "بدون حر",
        "عادي",
    ]

    nonspicy_flag = any(k in text for k in nonspicy_keywords_en + nonspicy_keywords_ar)

    spicy_keywords = [
        "spicy",
        "hot",
        "حار",
    ]
    spicy_flag = any(k in text for k in spicy_keywords) and not nonspicy_flag

    return spicy_flag, nonspicy_flag


def send_category_buttons(user_number: str, lang: str = "en", show_image: bool = True):
    """
    Jab user 'Order via text' ya 'Add more items' bole:

    - show_image=True  -> menu image + category buttons
    - show_image=False -> sirf category buttons (no image)
    """

    caption = "Here is our menu " if lang == "en" else "هذه قائمتنا"

    if show_image:
        image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
        try:
            send_whatsapp_image(user_number, image_url, caption=caption)
        except Exception as e:
            print("Error sending menu image in send_category_buttons:", e)

    if lang == "ar":
        body = "من فضلك اختر الفئة:"
        buttons = [
            {"id": "cat_burgers_meals", "title": "برجر ووجبات"},
            {"id": "cat_sandwiches", "title": "ساندويتش"},
            {"id": "cat_snacks_sides", "title": "وجبات خفيفة"},
        ]
    else:
        body = "Please choose a category:"
        buttons = [
            {"id": "cat_burgers_meals", "title": "Burgers & Meals"},
            {"id": "cat_sandwiches", "title": "Sandwiches & Wraps"},
            {"id": "cat_snacks_sides", "title": "Snacks & Sides"},
        ]

    send_whatsapp_quick_buttons(user_number, body, buttons)


def shorten_button_title(title: str, max_len: int = 20) -> str:
    """
    WhatsApp Cloud quick-reply button title must be 1–20 chars.
    If longer, truncate and add '...' so total <= max_len.
    """
    title = (title or "").strip()
    if len(title) <= max_len:
        return title
    # 17 chars + "..." = 20
    return title[: max_len - 3].rstrip() + "..."


def send_items_for_category(user_number: str, category: str, lang: str = "en"):
    items = get_items_by_category(category)
    if not items:
        send_whatsapp_text(user_number, "No items found in this category yet.")
        return

    state = WA_CATEGORY_STATE.get(user_number, {"category": category, "index": 0})
    index = state.get("index", 0)

    # if category changed, reset index
    if state.get("category") != category:
        index = 0

    slice_items = items[index : index + 2]
    buttons = []

    for item in slice_items:
        en_name = item["en"]
        ar_name = item["ar"]

        if lang == "ar":
            raw_title = ar_name
        else:
            raw_title = en_name.title()

        title = shorten_button_title(raw_title)  # 🔹 enforce <= 20 chars
        buttons.append({"id": f"item_{en_name}", "title": title})

    # "More items" button (already short enough)
    if index + 2 < len(items):
        if lang == "ar":
            more_title = "المزيد من الأصناف"
        else:
            more_title = "More items"
        buttons.append({"id": "more_items", "title": more_title})

    if lang == "ar":
        body = "من فضلك اختر الصنف (سيتم إضافة 1 كمية لكل ضغطة):"
    else:
        body = "Please choose an item (1 quantity will be added per click):"

    print("send_items_for_category buttons:", buttons)  # optional debug
    send_whatsapp_quick_buttons(user_number, body, buttons)

    WA_CATEGORY_STATE[user_number] = {"category": category, "index": index}


# ---------------------------
# LLM Reply
# ---------------------------


def get_llm_reply(msg, lang="en"):
    # If OpenRouter key missing/placeholder, just give a simple fallback message
    if not OPENROUTER_API_KEY or OPENROUTER_API_KEY.startswith("YOUR_"):
        return (
            "You can place your order by telling me the items and quantities."
            if lang == "en"
            else "يمكنك وضع طلبك بكتابة الأصناف والكمية."
        )

    lang_name = "English" if lang == "en" else "Arabic"

    sys_prompt = (
        "You are Joana Fast Food Assistant.\n"
        "- Short, clear, polite answers.\n"
        "- Can answer non-restaurant questions briefly, then bring user back to Joana.\n"
        "- Joana Fast Food is 24/7, takeaway only.\n"
        f"Always respond in {lang_name}.\n"
    )

    context = build_menu_context()

    messages = [
        {"role": "system", "content": sys_prompt},
        {"role": "system", "content": context},
    ]

    for m in session.get("messages", []):
        messages.append(m)

    messages.append({"role": "user", "content": msg})

    try:
        res = client.chat.completions.create(
            model="gpt-4o-mini",
            messages=messages,
            temperature=0.5,
            max_tokens=250,
        )
        return res.choices[0].message.content.strip()

    except Exception as e:
        print("LLM error:", e)
        return "Sorry, something went wrong." if lang == "en" else "عذراً، حدث خطأ ما."


# ---------------------------
# Routes
# ---------------------------


@app.route("/")
def index():
    return render_template("index.html")


@app.route("/cloud-test")
def cloud_test():
    # yahan apna khud ka number daal kar test kar sakti ho
    send_whatsapp_text("9230XXXXXXXX", "Hello from Joana bot via WhatsApp Cloud API!")
    return "Cloud test message sent!"


# ---------------------------
# WhatsApp Cloud Webhook
# ---------------------------

@app.route("/whatsapp/webhook", methods=["GET", "POST"])
def whatsapp_webhook():
    # 1) Verification handshake (GET)
    if request.method == "GET":
        mode = request.args.get("hub.mode")
        token = request.args.get("hub.verify_token")
        challenge = request.args.get("hub.challenge")

        print("Webhook verification:", mode, token, challenge)

        if mode == "subscribe" and token == WHATSAPP_VERIFY_TOKEN:
            return challenge or "", 200
        else:
            return "Verification token mismatch", 403

    # 2) Incoming messages (POST)
    data = request.get_json(force=True) or {}
    try:
        print("Incoming from Cloud API:", json.dumps(data, indent=2, ensure_ascii=False))
    except Exception:
        print("Incoming from Cloud API (raw):", data)

    user_text = None
    user_number = None
    btn_id = None  # 🔹 yahan id store karenge
    is_voice = False  # 🔊 track if this came from voice

    try:
        entry = (data.get("entry") or [])[0]
        changes = (entry.get("changes") or [])[0]
        value = changes.get("value") or {}
    except Exception:
        value = {}

    # Status callbacks (delivery/read) ignore
    if value.get("statuses"):
        print("Cloud API status callback – ignoring")
        return "ok", 200

    messages = value.get("messages")
    if not messages:
        print("No messages in payload.")
        return "ignored", 200

    msg_obj = messages[0]
    user_number = normalize_number(msg_obj.get("from"))
    msg_type = msg_obj.get("type")
    print("Cloud msg_type:", msg_type)

    if msg_type == "text":
        user_text = msg_obj.get("text", {}).get("body", "")

    elif msg_type == "interactive":
        interactive = msg_obj.get("interactive", {}) or {}
        itype = interactive.get("type")
        print("Interactive payload:", interactive)

        if itype == "button_reply":
            br = interactive.get("button_reply", {}) or {}
            btn_id = (br.get("id") or "").strip()
            user_text = (br.get("title") or btn_id or "").strip()
            print("Button reply => id:", btn_id, "| title:", user_text)
        elif itype == "list_reply":
            lr = interactive.get("list_reply", {}) or {}
            btn_id = (lr.get("id") or "").strip()
            user_text = (lr.get("title") or btn_id or "").strip()
            print("List reply => id:", btn_id, "| title:", user_text)

    # 🔊 NEW: handle voice / audio messages
    elif msg_type in ["audio", "voice"]:
        audio_obj = msg_obj.get("audio") or msg_obj.get("voice") or {}
        print("Cloud audio_obj:", audio_obj)
        transcribed = transcribe_audio_from_cloud(audio_obj)
        if not transcribed:
            # agar transcription fail ho jaye to polite message de do
            send_whatsapp_text(
                user_number,
                "Sorry, I couldn't understand the voice message. "
                "Please try again or type your order.",
            )
            return "ok", 200

        user_text = transcribed
        is_voice = True
        print("Voice converted to text:", repr(user_text))

    else:
        # other types (image, etc.) -> ignore or basic message
        user_text = ""

    if not user_number or not user_text:
        print("Could not parse user_number or user_text from Cloud payload.")
        return "ignored", 200

    print("Parsed RAW:", user_number, "| text:", repr(user_text), "| id:", repr(btn_id))
    if "\n" in user_text:
        parts = [p.strip() for p in user_text.split("\n") if p.strip()]
        if parts:
            user_text = parts[-1]
    print("Parsed CLEANED:", user_number, "| text:", repr(user_text))

    clean = user_text.strip().lower()
    clean_id = (btn_id or "").strip().lower()   # 🔹 id ka lowercase

    lang = detect_language(user_text or "")
    if any("\u0600" <= ch <= "\u06ff" for ch in user_text):
        lang = "ar"

    last_state = WHATSAPP_SESSIONS.get(user_number, {}).get("state", {}) or {}
    last_stage = last_state.get("stage")

    # 🔥 NEW: handle spicy / non-spicy buttons specially for WhatsApp
    if last_stage == "await_spice" and clean_id in ["spice_spicy", "spice_non_spicy"]:
        sess = WHATSAPP_SESSIONS.get(user_number, {})
        s = sess.get("state") or {}
        last_item = s.get("last_item")
        last_qty = s.get("last_qty", 1)

        if last_item:
            price, _category = get_price_and_category(last_item)

            # Ensure order list exists
            order_list = s.get("order")
            if not isinstance(order_list, list):
                order_list = []
            s["order"] = order_list

            if clean_id == "spice_spicy":
                spicy_q = last_qty
                non_q = 0
            else:  # spice_non_spicy
                spicy_q = 0
                non_q = last_qty

            if spicy_q > 0:
                order_list.append(
                    {
                        "item": last_item,
                        "qty": spicy_q,
                        "spicy": 1,
                        "nonspicy": 0,
                        "price": price,
                        "subtotal": spicy_q * price,
                    }
                )

            if non_q > 0:
                order_list.append(
                    {
                        "item": last_item,
                        "qty": non_q,
                        "spicy": 0,
                        "nonspicy": 1,
                        "price": price,
                        "subtotal": non_q * price,
                    }
                )

            # reset stage & last item, go to add_more
            s["stage"] = "add_more"
            s["last_item"] = None
            s["last_qty"] = 0
            sess["state"] = s
            WHATSAPP_SESSIONS[user_number] = sess

            # reply text
            if lang == "ar":
                kind = "حار" if clean_id == "spice_spicy" else "بدون حار"
                reply_text = (
                    f"تمت إضافة {last_item} {kind} ×{last_qty}! "
                    "هل ترغب في إضافة شيء آخر؟"
                )
                buttons = [
                    {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
                    {"id": "finish_order", "title": "إنهاء الطلب"},
                ]
            else:
                kind = "spicy" if clean_id == "spice_spicy" else "non-spicy"
                reply_text = (
                    f"Added {last_qty} {kind} {last_item.title()}! "
                    "Would you like to add anything else?"
                )
                buttons = [
                    {"id": "add_more_items", "title": "Add more items"},
                    {"id": "finish_order", "title": "Finish order"},
                ]

            send_whatsapp_quick_buttons(user_number, reply_text, buttons)
            return "ok", 200
        # if for some reason last_item missing, fall through to normal logic

    # 1️⃣ FIRST MESSAGE on WhatsApp -> always show branches + "Order via text / voice"
    if last_stage is None:
        if lang == "ar":
            buttons = [
                {"id": "order_text", "title": "الطلب عبر الرسائل"},
                {"id": "order_voice", "title": "الطلب عبر الصوت"},
            ]
            body = build_branches_message("ar")
        else:
            buttons = [
                {"id": "order_text", "title": "Order via text"},
                {"id": "order_voice", "title": "Order via voice"},
            ]
            body = build_branches_message("en")

        send_whatsapp_quick_buttons(user_number, body, buttons)
        # initialize empty session
        WHATSAPP_SESSIONS[user_number] = {"state": {"stage": "branch"}, "messages": []}
        return "ok", 200

    # 2️⃣ Category selection (title + id dono se match)
    if (
        clean in ["burgers & meals", "burgers_meals", "برجر ووجبات"]
        or clean_id == "cat_burgers_meals"
    ):
        WA_CATEGORY_STATE[user_number] = {"category": "burgers_meals", "index": 0}
        send_items_for_category(user_number, "burgers_meals", lang)
        return "ok", 200

    if (
        clean in ["sandwiches & wraps", "sandwiches", "ساندويتش", "ساندويتشات"]
        or clean_id == "cat_sandwiches"
    ):
        WA_CATEGORY_STATE[user_number] = {"category": "sandwiches", "index": 0}
        send_items_for_category(user_number, "sandwiches", lang)
        return "ok", 200

    if (
        clean in ["snacks & sides", "snacks_sides", "وجبات خفيفة"]
        or clean_id == "cat_snacks_sides"
    ):
        WA_CATEGORY_STATE[user_number] = {"category": "snacks_sides", "index": 0}
        send_items_for_category(user_number, "snacks_sides", lang)
        return "ok", 200

    # 3️⃣ More items in category (English + Arabic both)
    if (
        clean in ["more items", "المزيد من الأصناف"]           # exact button title
        or ("more" in clean and "item" in clean)                # user manually typed
        or clean_id == "more_items"                             # button id
    ):
        print("➡ MORE ITEMS clicked | clean =", clean, "| id =", clean_id)

        st = WA_CATEGORY_STATE.get(user_number, {})
        cat = st.get("category")
        idx = st.get("index", 0)

        if not cat:
            print("⚠ No category saved for this user in WA_CATEGORY_STATE")
        else:
            items = get_items_by_category(cat)
            if items:
                new_index = idx + 2
                if new_index >= len(items):
                    # wrap to start
                    new_index = 0

                WA_CATEGORY_STATE[user_number] = {
                    "category": cat,
                    "index": new_index,
                }

                print(
                    f"➡ Paging items for category={cat}, "
                    f"old_index={idx}, new_index={new_index}, total_items={len(items)}"
                )

                send_items_for_category(user_number, cat, lang)
                return "ok", 200

    # 4️⃣ Add more / finish order buttons (from interactive)
    if (
        clean in ["add more items", "إضافة أصناف أخرى", "اضافة اصناف اخرى"]
        or clean_id in ["add_more_items", "payment_add_more"]
    ) and last_stage in ["add_more", "payment"]:
        sess = WHATSAPP_SESSIONS.get(user_number)
        if isinstance(sess, dict):
            st = sess.get("state") or {}
            st["stage"] = "add_more"
            sess["state"] = st
            WHATSAPP_SESSIONS[user_number] = sess

        send_category_buttons(user_number, lang, show_image=False)
        return "ok", 200

    if clean in ["finish order", "إنهاء الطلب", "انهاء الطلب"] or clean_id == "finish_order":
        user_text = "done"
        clean = "done"

    # 5️⃣ "Order via text" button
    if clean_id == "order_text" or clean in [
        "1",
        "1.",
        "order via text",
        "order by text",
        "text order",
        "الطلب عبر الرسائل",
        "الطلب بالرسائل",
        "اطلب بالرسائل",
    ]:
        send_category_buttons(user_number, lang, show_image=True)
        return "ok", 200

    # 6️⃣ "Order via voice"
    if clean_id == "order_voice" or clean in [
        "2",
        "2.",
        "order via voice",
        "order by voice",
        "voice order",
        "voice",
        "الطلب عبر الصوت",
        "الطلب بالصوت",
        "اطلب بالصوت",
    ]:
        send_whatsapp_text(
            user_number,
            "🎙 من فضلك أرسل طلبك على شكل رسالة صوتية.\n\n"
            'مثال: "اثنين زنجر برجر، واحد حار وواحد بدون حار، وبطاطس كبيرة".\n\n'
            "سيقوم الموظف بالاستماع وتأكيد طلبك 😊",
        )
        return "ok", 200

    # 7️⃣ Baaki sab normal chatbot flow (aap ka purana code yahan se same)
    try:
        result = process_whatsapp_message(user_number, user_text, lang_hint=lang, is_voice=is_voice)

        reply_html = result.get("reply") or "Sorry, something went wrong."
        reply_text = html_to_whatsapp(reply_html)

        stage = result.get("stage")
        order = result.get("order") or []

        if result.get("menu"):
            image_url = (
                "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
            )
            send_whatsapp_image(user_number, image_url, caption="Here is our menu 📋")

        # Spicy selection buttons
        if stage == "await_spice":
            if lang == "ar":
                buttons = [
                    {"id": "spice_spicy", "title": "حار"},
                    {"id": "spice_non_spicy", "title": "بدون حار"},
                ]
            else:
                buttons = [
                    {"id": "spice_spicy", "title": "Spicy"},
                    {"id": "spice_non_spicy", "title": "Non-spicy"},
                ]
            send_whatsapp_quick_buttons(user_number, reply_text, buttons)
            return "ok", 200

        # Add more / finish order buttons (after adding an item)
        if stage == "add_more" and order:
            if lang == "ar":
                buttons = [
                    {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
                    {"id": "finish_order", "title": "إنهاء الطلب"},
                ]
            else:
                buttons = [
                    {"id": "add_more_items", "title": "Add more items"},
                    {"id": "finish_order", "title": "Finish order"},
                ]
            send_whatsapp_quick_buttons(user_number, reply_text, buttons)
            return "ok", 200

        # Payment page buttons
        if stage == "payment" and order:
            if lang == "ar":
                buttons = [
                    {"id": "payment_proceed", "title": "المتابعة إلى الدفع"},
                    {"id": "payment_add_more", "title": "إضافة أصناف أخرى"},
                ]
            else:
                buttons = [
                    {"id": "payment_proceed", "title": "Proceed to payment"},
                    {"id": "payment_add_more", "title": "Add more items"},
                ]
            send_whatsapp_quick_buttons(user_number, reply_text, buttons)
            return "ok", 200

        # Choose payment method buttons
        if stage == "choose_payment":
            if lang == "ar":
                buttons = [
                    {"id": "pay_online", "title": "الدفع عبر الإنترنت"},
                    {"id": "pay_cash", "title": "الدفع نقداً"},
                ]
            else:
                buttons = [
                    {"id": "pay_online", "title": "Online payment"},
                    {"id": "pay_cash", "title": "Cash"},
                ]
            send_whatsapp_quick_buttons(user_number, reply_text, buttons)
            return "ok", 200

    except Exception as e:
        print("Error in whatsapp_webhook:", e)
        reply_text = "Sorry, something went wrong."

    send_whatsapp_text(user_number, reply_text)
    return "ok", 200


def process_whatsapp_message(
    user_number: str, user_text: str, lang_hint: str = None, is_voice: bool = False
) -> dict:
    ctx = WHATSAPP_SESSIONS.get(user_number, {})
    prev_state = ctx.get("state")
    prev_messages = ctx.get("messages")

    with app.test_request_context(
        "/api/chat",
        method="POST",
        json={
            "message": user_text,
            "is_voice": is_voice,   # 🔊 forward flag to chat()
            "lang_hint": lang_hint,
        },
    ):
        if prev_state is not None:
            session["state"] = prev_state
        if prev_messages is not None:
            session["messages"] = prev_messages

        resp = chat()

        WHATSAPP_SESSIONS[user_number] = {
            "state": session.get("state"),
            "messages": session.get("messages"),
        }

        if isinstance(resp, tuple):
            resp_obj = resp[0]
        else:
            resp_obj = resp

        data = resp_obj.get_json() or {}
        return data


# ---------------------------
# Web chat route
# ---------------------------


def make_chat_response(reply, lang, menu=None):
    s = session.get("state", {"stage": None, "order": [], "total": 0})
    payload = {
        "reply": reply,
        "lang": lang,
        "stage": s.get("stage"),
        "order": s.get("order"),
        "total": s.get("total", 0),
    }
    if menu:
        payload["menu"] = menu
    return jsonify(payload)


@app.route("/api/chat", methods=["POST"])
def chat():
    global MENU, BRANCHES
    MENU = load_menu()
    BRANCHES = load_branches()

    s = session.get("state", {"stage": None, "order": [], "total": 0})
    session["state"] = s
    session["messages"] = session.get("messages", [])

    data = request.get_json(force=True)
    msg_raw = (data.get("message") or "").strip()
    msg = msg_raw
    is_voice = data.get("is_voice", False)
    lang_hint = data.get("lang_hint")

    lang = detect_language(msg_raw)

    if any("\u0600" <= ch <= "\u06ff" for ch in msg_raw):
        lang = "ar"

    if lang_hint in ("en", "ar"):
        lang = lang_hint

    if lang == "ar":
        msg = llm_normalize_text(msg_raw, lang)

    intent = detect_intent(msg)

    session["messages"].append({"role": "user", "content": msg})

    msg_l = msg.lower()
    if "menu" in msg_l or "send menu" in msg_l or "price list" in msg_l:
        intent = "menu"

    if polite_check(msg):
        reply = "Please speak politely." if lang == "en" else "من فضلك تحدث بأدب."
        if is_voice:
            speak_text(reply, lang)
        return make_chat_response(reply, lang)

    AR_GREETINGS = [
        "مرحبا",
        "مرحبا بك",
        "اهلا",
        "أهلا",
        "اهلاً",
        "اهلا وسهلا",
        "السلام عليكم",
        "السلام",
        "هلا",
        "يا هلا",
        "مساء الخير",
        "صباح الخير",
    ]

    EN_GREETINGS = [
        "hi",
        "hii",   # NEW
        "hello",
        "hey",
        "good morning",
        "good evening",
        "good afternoon",
        "hiya",
        "yo",
    ]

    def is_greeting_text(msg_: str, lang_: str) -> bool:
        """
        Sirf tab greeting ho jab pure text me sirf greeting hi ho.
        'chicken sandwich' jese text ko greeting na samjho.
        """
        if not msg_:
            return False

        text = msg_.strip().lower()
        text = re.sub(r"[^\w\u0600-\u06FF ]+", "", text)

        greetings = AR_GREETINGS if lang_ == "ar" else EN_GREETINGS
        return text in greetings

    # Greeting (only if stage empty) – for web chat
    if s.get("stage") is None and is_greeting_text(msg, lang):
        s = {
            "stage": "branch",
            "order": [],
            "total": 0,
            "last_item": None,
            "last_qty": 0,
        }
        session["state"] = s

        if lang == "ar":
            reply = (
                "مرحباً بك في مطعم جوانا للوجبات السريعة! 🍔<br>"
                "ابحث عن أقرب فرع واتصل على الرقم للطلب:<br><br>"
                + "<br>".join(
                    [
                        (
                            f"<b>{b.get('Branch Name','')}</b> – "
                            f"{b.get('Address / Area','')}<br>"
                            f"📞 <a href='tel:{b.get('Phone Number','')}'>"
                            f"<b>{b.get('Phone Number','')}</b></a><br>"
                        )
                        for b in BRANCHES
                        if b.get("Phone Number")
                    ][:6]
                )
                + "<br>جاهز للطلب؟ شارك طلبك بالصوت أو بالكتابة!<br><br>"
            )
        else:
            reply = (
                "Welcome to JOANA Fast Food! <br>"
                "Find your nearest branch and dial the number to order:<br><br>"
                + "<br>".join(
                    [
                        (
                            f"<b>{b.get('Branch Name','')}</b> – "
                            f"{b.get('Address / Area','')}<br>"
                            f"📞 <a href='tel:{b.get('Phone Number','')}'>"
                            f"<b>{b.get('Phone Number','')}</b></a><br>"
                        )
                        for b in BRANCHES
                        if b.get("Phone Number")
                    ][:6]
                )
                + "<br>Ready to order? Share your order via voice or chat!<br><br>"
            )

        if is_voice:
            speak_text(reply, lang)

        return make_chat_response(reply, lang)

    if intent == "menu":
        reply = (
            "Here’s our menu! Please place your order."
            if lang == "en"
            else "هذه قائمتنا! من فضلك ضع طلبك."
        )
        if is_voice:
            speak_text(reply, lang)
        return make_chat_response(reply, lang, menu="/static/menu.PNG")

    # user says "add more" in payment (free text)
    if s.get("stage") == "payment" and (
        "add more" in msg.lower() or "إضافة" in msg or "اضافة" in msg or "أخرى" in msg
    ):
        s["stage"] = "add_more"
        session["state"] = s
        reply = (
            "Sure! Please tell me what else you would like to add to your order."
            if lang == "en"
            else "حسناً! من فضلك أخبرني ما الذي تريد إضافته أيضاً."
        )
        if is_voice:
            speak_text(reply, lang)
        return make_chat_response(reply, lang)

    # Spicy follow-up
    if s.get("stage") == "await_spice" and s.get("last_item"):
        last_item = s["last_item"]
        last_qty = s.get("last_qty", 1)
        price, _category = get_price_and_category(last_item)
        split = parse_spice_split(msg)

        if split:
            spicy_q, non_q = split
        else:
            spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

            if spicy_flag:
                spicy_q, non_q = last_qty, 0
            elif nonspicy_flag:
                spicy_q, non_q = 0, last_qty
            else:
                spicy_q, non_q = 0, last_qty

        order_added_msgs = []

        if spicy_q > 0:
            s["order"].append(
                {
                    "item": last_item,
                    "qty": spicy_q,
                    "spicy": 1,
                    "nonspicy": 0,
                    "price": price,
                    "subtotal": spicy_q * price,
                }
            )
            order_added_msgs.append(f"{spicy_q} {'spicy' if lang == 'en' else 'حار'}")

        if non_q > 0:
            s["order"].append(
                {
                    "item": last_item,
                    "qty": non_q,
                    "spicy": 0,
                    "nonspicy": 1,
                    "price": price,
                    "subtotal": non_q * price,
                }
            )
            order_added_msgs.append(
                f"{non_q} {'non-spicy' if lang == 'en' else 'بدون حار'}"
            )

        s["stage"] = "add_more"
        s["last_item"] = None
        s["last_qty"] = 0
        session["state"] = s

        reply = (
            f"Added {last_item.title()} — "
            + " & ".join(order_added_msgs)
            + ". Would you like to add anything else?"
            if lang == "en"
            else f"تمت إضافة {last_item} — "
            + " و ".join(order_added_msgs)
            + ". هل ترغب في إضافة شيء آخر؟"
        )

        if is_voice:
            speak_text(reply, lang)

        return make_chat_response(reply, lang)

    # Item detection
    found = find_menu_item(msg)

    if found:
        qty = detect_qty(msg)
        price, category = get_price_and_category(found)

        s["last_item"] = found
        s["last_qty"] = qty
        session["state"] = s

        spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

        if category == "burgers_meals" and not (spicy_flag or nonspicy_flag):
            s["stage"] = "await_spice"
            session["state"] = s

            reply = (
                f"Would you like your {found.title()} spicy or non-spicy?"
                if lang == "en"
                else f"هل ترغب أن يكون {found} حارًا أم بدون حار؟"
            )

            if is_voice:
                speak_text(reply, lang)

            return make_chat_response(reply, lang)

        if category == "burgers_meals":
            if spicy_flag:
                spicy = 1
                nonspicy = 0
            elif nonspicy_flag:
                spicy = 0
                nonspicy = 1
            else:
                spicy = 0
                nonspicy = 1
        else:
            spicy = 0
            nonspicy = 0

        s["order"].append(
            {
                "item": found,
                "qty": qty,
                "spicy": spicy,
                "nonspicy": nonspicy,
                "price": price,
                "subtotal": qty * price,
            }
        )

        previous_stage = s.get("stage")
        s["last_item"] = None
        s["last_qty"] = 0

        if previous_stage == "payment":
            summary, total = build_order_summary_and_total(s["order"], lang)
            s["stage"] = "payment"
            s["total"] = total
            session["state"] = s

            reply = (
                "Your order has been updated! Here's the new summary:<br>"
                + "<br>".join(summary)
                + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
                + "Would you like to proceed with the payment?"
                if lang == "en"
                else "تم تحديث طلبك!<br>"
                + "<br>".join(summary)
                + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
                + "هل ترغب في المتابعة إلى الدفع؟"
            )
        else:
            s["stage"] = "add_more"
            session["state"] = s

            reply = (
                f"{found.title()} ×{qty} added! Would you like to add anything else?"
                if lang == "en"
                else f"تمت إضافة {found} ×{qty}! هل ترغب في إضافة شيء آخر؟"
            )

        if is_voice:
            speak_text(reply, lang)

        return make_chat_response(reply, lang)

    # Done / no more items
    if any(x in msg.lower() for x in ["no", "done", "that's all", "that all"]) or any(
        x in msg for x in ["خلص", "لا", "انتهى", "كفاية"]
    ):
        if s.get("order"):

            summary, total = build_order_summary_and_total(s["order"], lang)

            reply = (
                "Your order is confirmed! Here's a summary:<br>"
                + "<br>".join(summary)
                + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
                + "Would you like to proceed with the payment?"
                if lang == "en"
                else "تم تأكيد طلبك!<br>"
                + "<br>".join(summary)
                + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
                + "هل ترغب في المتابعة إلى الدفع؟"
            )

            s["stage"] = "payment"
            s["total"] = total
            session["state"] = s

            if is_voice:
                speak_text(reply, lang)

            return make_chat_response(reply, lang)

    # Payment yes
    if s.get("stage") == "payment" and (
        any(
            x in msg.lower()
            for x in [
                "yes",
                "sure",
                "ok",
                "okay",
                "proceed",
                "go ahead",
            ]
        )
        or any(
            x in msg
            for x in [
                "تمام",
                "نعم",
                "أكيد",
                "المتابعة",
                "تابع",
                "ادفع",
                "إلى الدفع",
            ]
        )
    ):
        reply = (
            "How would you like to pay?<br><b>Cash</b> or <b>Online Payment</b>?"
            if lang == "en"
            else "كيف ترغب في الدفع؟<br><b> نقدي </b> أو <b> الدفع عبر الإنترنت</b>؟"
        )
        s["stage"] = "choose_payment"
        session["state"] = s

        if is_voice:
            speak_text(reply, lang)

        return make_chat_response(reply, lang)

    # Choose payment
    if s.get("stage") == "choose_payment":
        total = s.get("total", 0)
        msg_l = msg.lower()

        if any(x in msg_l for x in ["cash", "cod", "نقد", "نقدا"]):
            reply = (
                "Thank you for your order! Your order will be ready in 20 to 30 minutes."
                if lang == "en"
                else "شكراً لطلبك! سيكون جاهزاً خلال 20 إلى 30 دقيقة."
            )
            s = {
                "stage": None,
                "order": [],
                "total": 0,
                "last_item": None,
                "last_qty": 0,
            }
            session["state"] = s

            if is_voice:
                speak_text(reply, lang)

            return make_chat_response(reply, lang)

        elif any(
            x in msg_l
            for x in [
                "online",
                "online payment",
                "pay online",
                "card",
                "visa",
                "master",
                "debit",
                "mada",
            ]
        ) or any(x in msg for x in ["مدى", "إلكتروني", "دفع", "كرت"]):
            reply = (
                f"Great! Your payment of {total:.2f} {CURRENCY} is being processed.<br>"
                f"<a href='{PAYMENT_URL}' target='_blank'>"
                f"<b>Click here to complete your payment</b></a>"
                if lang == "en"
                else f"رائع! يتم الآن معالجة دفعتك بمقدار {total:.2f} {CURRENCY_AR}.<br>"
                f"<a href='{PAYMENT_URL}' target='_blank'>"
                f"<b>اضغط هنا لإتمام الدفع</b></a>"
            )
            s = {
                "stage": None,
                "order": [],
                "total": 0,
                "last_item": None,
                "last_qty": 0,
            }
            session["state"] = s

            if is_voice:
                speak_text(reply, lang)

            return make_chat_response(reply, lang)

        else:
            reply = (
                "Please select a valid payment method — Cash or Online."
                if lang == "en"
                else "يرجى اختيار طريقة الدفع الصحيحة - نقدًا أو عبر الإنترنت"
            )
            if is_voice:
                speak_text(reply, lang)

            return make_chat_response(reply, lang)

    # Generic LLM reply
    reply = get_llm_reply(msg, lang)

    if s.get("stage") == "payment" and s.get("order"):
        summary, total = build_order_summary_and_total(s["order"], lang)
        s["total"] = total
        session["state"] = s

        if lang == "en":
            reply = (
                reply
                + "<br><br>Here is your current order summary:<br>"
                + "<br>".join(summary)
                + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
                + "Would you like to proceed with the payment?"
            )
        else:
            reply = (
                reply
                + "<br><br>هذا ملخص طلبك الحالي:<br>"
                + "<br>".join(summary)
                + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
                + "هل ترغب في المتابعة إلى الدفع؟"
            )

    if is_voice:
        speak_text(reply, lang)

    return make_chat_response(reply, lang)


# ---------------------------
# Run App (local)
# ---------------------------
if __name__ == "__main__":
    app.run(debug=True)




# English text and arabic text working perfectly
# import json
# import os
# import re
# import sqlite3
# import requests

# import pandas as pd
# from flask import Flask, render_template, request, jsonify, session
# from nlp_utils import detect_intent, detect_language
# from openai import OpenAI
# from dotenv import load_dotenv

# load_dotenv()

# # ---------------------------
# # ENV VARIABLES
# # ---------------------------
# OPENROUTER_API_KEY = os.getenv("OPENROUTER_API_KEY", "YOUR_OPENROUTER_API_KEY_HERE")
# FLASK_SECRET = os.getenv("FLASK_SECRET_KEY", "joana-fastfood-secret")

# # WhatsApp Cloud API envs
# WHATSAPP_TOKEN = os.getenv("WHATSAPP_TOKEN")
# WHATSAPP_PHONE_NUMBER_ID = os.getenv("WHATSAPP_PHONE_NUMBER_ID")
# WHATSAPP_VERIFY_TOKEN = os.getenv("WHATSAPP_VERIFY_TOKEN", "joana-verify-token-123")

# WHATSAPP_API_BASE = "https://graph.facebook.com/v21.0"

# # ---------------------------
# # WHATSAPP CLOUD SEND HELPERS
# # ---------------------------


# def send_whatsapp_text(to_number: str, text: str):
#     """
#     Send plain text via WhatsApp Cloud API
#     """
#     print("=== send_whatsapp_text CALLED ===")
#     print("WHATSAPP_TOKEN present?", bool(WHATSAPP_TOKEN))
#     print("Sending to:", to_number, "| Text:", text)

#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "text",
#         "text": {"body": text},
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD SEND RESPONSE:", resp.status_code, resp.text)
#     except Exception as e:
#         print("Error sending WhatsApp text via Cloud API:", repr(e))


# def send_whatsapp_image(to_number: str, image_url: str, caption: str = ""):
#     """
#     Send image + optional caption via WhatsApp Cloud API
#     """
#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "image",
#         "image": {
#             "link": image_url,
#         },
#     }
#     if caption:
#         payload["image"]["caption"] = caption

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD IMAGE SEND:", resp.status_code, resp.text)
#         return resp.status_code in (200, 201)
#     except Exception as e:
#         print("Error sending WhatsApp image via Cloud API:", e)
#         return False


# def send_whatsapp_quick_buttons(to_number: str, body_text: str, buttons: list):
#     """
#     Send interactive "button" template via WhatsApp Cloud API
#     (max 3 buttons)
#     """
#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }

#     # Cloud API button format
#     wa_buttons = []
#     for b in buttons[:3]:
#         wa_buttons.append(
#             {
#                 "type": "reply",
#                 "reply": {
#                     "id": b.get("id", b["title"]),
#                     "title": b["title"],
#                 },
#             }
#         )

#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "interactive",
#         "interactive": {
#             "type": "button",
#             "body": {"text": body_text},
#             "action": {"buttons": wa_buttons},
#         },
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD BUTTONS SEND:", resp.status_code, resp.text)
#         return resp.status_code in (200, 201)
#     except Exception as e:
#         print("Error sending WhatsApp buttons via Cloud API:", e)
#         return False


# # ---------------------------
# # Paths / Directories
# # ---------------------------
# BASE_DIR = os.path.dirname(os.path.abspath(__file__))
# DATA_DIR = os.path.join(BASE_DIR, "data")

# MENU_FILE = os.path.join(DATA_DIR, "Menu.xlsx")
# BRANCHES_FILE = os.path.join(DATA_DIR, "Branches.xlsx")

# WHATSAPP_SESSIONS = {}
# WA_CATEGORY_STATE = {}  # phone -> {"category": str, "index": int}

# # ---------------------------
# # Flask Config
# # ---------------------------
# app = Flask(__name__, static_folder="static", template_folder="templates")
# app.secret_key = FLASK_SECRET

# client = OpenAI(api_key=OPENROUTER_API_KEY, base_url="https://openrouter.ai/api/v1")

# PAYMENT_URL = (
#     "https://starlit-sopapillas-520aa2.netlify.app/"
#     "?redirect=http://127.0.0.1:5000/thankyou"
# )
# CURRENCY = "SAR"
# CURRENCY_AR = "ريال سعودي"

# # ---------------------------
# # Database
# # ---------------------------
# DB_PATH = os.path.join(DATA_DIR, "orders.db")


# def init_db():
#     os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
#     conn = sqlite3.connect(DB_PATH)
#     conn.execute(
#         """
#         CREATE TABLE IF NOT EXISTS orders (
#             id INTEGER PRIMARY KEY AUTOINCREMENT,
#             item TEXT,
#             qty INTEGER,
#             spicy INTEGER,
#             nonspicy INTEGER,
#             price REAL,
#             subtotal REAL,
#             total REAL,
#             payment_method TEXT,
#             datetime TEXT
#         )
#         """
#     )
#     conn.commit()
#     conn.close()


# init_db()

# # ---------------------------
# # Category mapping helper
# # ---------------------------


# def canonical_category(raw_category: str, name_en: str = "") -> str:
#     """
#     Excel category ko 3 main groups me convert:
#     burgers_meals, sandwiches, snacks_sides
#     """
#     c = (raw_category or "").strip().lower()
#     n = (name_en or "").strip().lower()

#     if c in ["burgers", "burger", "meals", "meal"]:
#         return "burgers_meals"
#     if c == "sandwiches":
#         return "sandwiches"
#     if c in ["sides", "side", "drinks", "drink"]:
#         return "snacks_sides"

#     if any(
#         k in n for k in ["burger", "zinger", "tortilla", "hot dog", "hotdog", "meal"]
#     ):
#         return "burgers_meals"
#     if any(k in n for k in ["sandwich", "kudu", "kabab"]):
#         return "sandwiches"
#     if any(
#         k in n
#         for k in [
#             "fries",
#             "nuggets",
#             "rings",
#             "juice",
#             "corn",
#             "potato",
#             "samosa",
#             "popcorn",
#         ]
#     ):
#         return "snacks_sides"

#     return ""


# # ---------------------------
# # Excel Loaders
# # ---------------------------


# def load_menu():
#     try:
#         df_raw = pd.read_excel(MENU_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             if any("name_en" in str(c).lower() for c in row):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             print("❌ Could not find header row containing 'name_en'")
#             return {}

#         df = pd.read_excel(MENU_FILE, header=header_row_index)

#         def has(col, *keys):
#             c = str(col).strip().lower()
#             return any(k in c for k in keys)

#         english_col = next((c for c in df.columns if has(c, "name_en")), None)
#         arabic_col = next((c for c in df.columns if has(c, "name_ar")), None)
#         price_col = next((c for c in df.columns if has(c, "price")), None)
#         cat_col = next(
#             (c for c in df.columns if has(c, "category", "cat", "type")), None
#         )

#         if not english_col or not price_col:
#             raise Exception("Missing name_en or price column")

#         menu = {}
#         for _, row in df.iterrows():
#             en = str(row[english_col]).strip().lower()
#             if not en or en == "nan":
#                 continue

#             ar = str(row[arabic_col]).strip() if arabic_col else en
#             raw_cat = str(row.get(cat_col, "")).strip() if cat_col else ""

#             try:
#                 price = float(row[price_col])
#             except Exception:
#                 continue

#             cat_final = canonical_category(raw_cat, en)

#             entry = {
#                 "price": price,
#                 "category": cat_final,
#                 "name_en": en,
#                 "name_ar": ar or en,
#             }

#             menu[en] = entry
#             if ar:
#                 menu[ar.lower()] = entry

#         print(f"✅ Loaded {len(menu)} menu items.")
#         return menu

#     except Exception as e:
#         print("Menu load failed:", e)
#         return {}


# def load_branches():
#     try:
#         df_raw = pd.read_excel(BRANCHES_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             row_l = [str(c).lower() for c in row]
#             if any("branch" in c for c in row_l) and any("address" in c for c in row_l):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             header_row_index = 0

#         df = pd.read_excel(BRANCHES_FILE, header=header_row_index)

#         name_col = next((c for c in df.columns if "branch" in str(c).lower()), None)
#         addr_col = next((c for c in df.columns if "address" in str(c).lower()), None)
#         phone_col = next(
#             (
#                 c
#                 for c in df.columns
#                 if "phone" in str(c).lower() or "number" in str(c).lower()
#             ),
#             None,
#         )

#         branches = []
#         for _, row in df.iterrows():
#             branches.append(
#                 {
#                     "Branch Name": str(row.get(name_col, "")).strip(),
#                     "Address / Area": str(row.get(addr_col, "")).strip(),
#                     "Phone Number": str(row.get(phone_col, "")).strip(),
#                 }
#             )

#         print(f"✅ Loaded {len(branches)} branches.")
#         return [
#             b
#             for b in branches
#             if (b["Branch Name"] or b["Address / Area"] or b["Phone Number"])
#         ]

#     except Exception as e:
#         print("❌ Branch load failed:", e)
#         return []


# MENU = load_menu()
# BRANCHES = load_branches()

# # ---------------------------
# # Helpers
# # ---------------------------


# def build_menu_context():
#     if not MENU:
#         return "Current restaurant menu is empty."

#     lines = []
#     for name, info in MENU.items():
#         if not re.search(r"[A-Za-z]", name):
#             continue
#         price = info.get("price", 0.0)
#         category = info.get("category", "") or ""
#         lines.append(f"- {name} | price: {price:.2f} {CURRENCY} | category: {category}")

#     if not lines:
#         return "Current restaurant menu is empty."

#     return "Current restaurant menu (items, prices, categories):\n" + "\n".join(lines)


# def build_branches_message(lang: str = "en") -> str:
#     lines = []
#     for b in BRANCHES[:6]:
#         name = b.get("Branch Name", "")
#         addr = b.get("Address / Area", "")
#         phone = b.get("Phone Number", "")
#         if phone:
#             lines.append(f"- {name} – {addr} | {phone}")

#     branches_text = "\n".join(lines) if lines else "Branches info not available."

#     if lang == "ar":
#         return (
#             "مرحباً بك في مطعم جوانا للوجبات السريعة!\n\n"
#             "الفروع:\n"
#             f"{branches_text}\n\n"
#             "من فضلك اختر طريقة الطلب:"
#         )
#     else:
#         return (
#             "Welcome to JOANA Fast Food! \n\n"
#             "Branches:\n"
#             f"{branches_text}\n\n"
#             "Please choose an option:"
#         )


# def html_to_whatsapp(text: str) -> str:
#     if not text:
#         return ""

#     text = text.replace("<br>", "\n").replace("<br/>", "\n").replace("<br />", "\n")

#     def replace_anchor(match):
#         full_tag = match.group(0)
#         label = match.group(1).strip()

#         href_match = re.search(
#             r'href=[\'"]([^\'"]+)[\'"]', full_tag, flags=re.IGNORECASE
#         )
#         if not href_match:
#             return label

#         url = href_match.group(1).strip()

#         if url.startswith("tel:"):
#             return label

#         return f"{label}: {url}"

#     text = re.sub(
#         r"<a[^>]*>(.*?)</a>", replace_anchor, text, flags=re.IGNORECASE | re.DOTALL
#     )
#     text = re.sub(r"<[^>]+>", "", text)
#     return text.strip()


# def normalize_number(raw: str) -> str:
#     if not raw:
#         return ""
#     raw = str(raw)
#     if raw.startswith("+"):
#         raw = raw[1:]
#     return raw


# WA_GREETINGS = [
#     "hi",
#     "hii",  # NEW
#     "hello",
#     "hey",
#     "salam",
#     "slam",
#     "asalam",
#     "assalam",
#     "assalam o alaikum",
#     "assalamu alaikum",
#     "السلام عليكم",
#     "مرحبا",
# ]


# def is_wa_greeting(text: str) -> bool:
#     """
#     WhatsApp greeting sirf tab jab pura text hi greeting ho.
#     'chicken sandwich' mein 'hi' aata hai lekin yeh greeting nahi hai.
#     """
#     if not text:
#         return False
#     t = text.strip().lower()
#     t = re.sub(r"[^\w\u0600-\u06FF ]+", "", t)
#     return t in WA_GREETINGS


# def llm_normalize_text(text: str, lang: str) -> str:
#     if not text or lang != "ar":
#         return text
#     # If OpenRouter key invalid / missing, skip LLM normalization
#     if not OPENROUTER_API_KEY or OPENROUTER_API_KEY.startswith("YOUR_"):
#         return text
#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=[
#                 {
#                     "role": "system",
#                     "content": (
#                         "You are a spell-correction engine for user messages.\n"
#                         "- Fix Arabic spelling, don't translate.\n"
#                         "- Return only corrected text."
#                     ),
#                 },
#                 {"role": "user", "content": text},
#             ],
#             temperature=0,
#             max_tokens=100,
#         )
#         return res.choices[0].message.content.strip()
#     except Exception as e:
#         print("LLM normalize error:", e)
#         return text


# def detect_qty(msg: str) -> int:
#     if not msg:
#         return 1
#     text = msg.lower().strip()

#     arabic_digit_map = {
#         "٠": "0",
#         "١": "1",
#         "٢": "2",
#         "٣": "3",
#         "٤": "4",
#         "٥": "5",
#         "٦": "6",
#         "٧": "7",
#         "٨": "8",
#         "٩": "9",
#     }
#     for ar, en in arabic_digit_map.items():
#         text = text.replace(ar, en)

#     digit_match = re.search(r"\b(\d+)\b", text)
#     if digit_match:
#         try:
#             q = int(digit_match.group(1))
#             if q > 0:
#                 return q
#         except ValueError:
#             pass

#     number_words_en = {
#         "one": 1,
#         "two": 2,
#         "three": 3,
#         "four": 4,
#         "five": 5,
#         "six": 6,
#         "seven": 7,
#         "eight": 8,
#         "nine": 9,
#         "ten": 10,
#     }
#     for word, value in number_words_en.items():
#         if re.search(r"\b" + re.escape(word) + r"\b", text):
#             return value

#     number_words_ar = {
#         "واحد": 1,
#         "واحدة": 1,
#         "اثنين": 2,
#         "اتنين": 2,
#         "ثنين": 2,
#         "ثلاثة": 3,
#         "ثلاث": 3,
#         "اربعة": 4,
#         "أربعة": 4,
#         "خمسة": 5,
#         "ستة": 6,
#         "سبعة": 7,
#         "ثمانية": 8,
#         "تسعة": 9,
#         "عشرة": 10,
#     }
#     for word, value in number_words_ar.items():
#         if word in text:
#             return value

#     return 1


# def polite_check(text):
#     bad = ["idiot", "stupid", "حرام", "لعنة", "غبي"]
#     return any(w in text.lower() for w in bad)


# def get_price_and_category(name):
#     entry = MENU.get(name.lower(), {})
#     return float(entry.get("price", 0)), entry.get("category", "")


# def build_order_summary_and_total(order_items, lang):
#     total = sum(i.get("subtotal", 0) for i in order_items)
#     summary = []

#     for i in order_items:
#         item_name = i["item"]
#         item_info = MENU.get(item_name.lower(), {})
#         category = item_info.get("category")

#         if category == "burgers_meals":
#             if lang == "ar":
#                 kind = "حار" if i.get("spicy") else "بدون حار"
#             else:
#                 kind = "spicy" if i.get("spicy") else "non-spicy"
#             summary.append(f"{i['qty']} {kind} {item_name.title()}")
#         else:
#             summary.append(f"{i['qty']} {item_name.title()}")

#     return summary, total


# def speak_text(reply, lang):
#     return


# def parse_spice_split(msg):
#     text = msg.lower().replace("-", " ")
#     nums = re.findall(r"\d+", text)

#     if len(nums) >= 2:
#         n1, n2 = int(nums[0]), int(nums[1])

#         if ("spicy" in text or "حار" in text) and (
#             "non" in text or "mild" in text or "عادي" in text or "بدون" in text
#         ):
#             spicy_first = (
#                 text.find("spicy") < text.find("non") if "non" in text else True
#             )
#             arab_spicy_first = (
#                 text.find("حار") < text.find("عادي") if "عادي" in text else True
#             )

#             return (n1, n2) if (spicy_first or arab_spicy_first) else (n2, n1)

#     return None


# def find_menu_item(msg):
#     text = msg.lower()
#     candidates = []

#     for name in MENU.keys():
#         pattern = r"\b" + re.escape(name) + r"s?\b"
#         if re.search(pattern, text):
#             candidates.append(name)

#     if not candidates:
#         return None

#     candidates.sort(key=len, reverse=True)
#     return candidates[0]


# def get_items_by_category(category: str):
#     """
#     Return unique items for a category:
#     [{"en": "...", "ar": "..."}]
#     """
#     items_map = {}
#     for key, info in MENU.items():
#         if info.get("category") != category:
#             continue
#         en = (info.get("name_en") or key).strip().lower()
#         ar = (info.get("name_ar") or en).strip()
#         items_map[en] = {"en": en, "ar": ar}
#     return list(items_map.values())


# def detect_spicy_nonspicy(msg: str):
#     """
#     Text se spicy / non-spicy flags nikalta hai.
#     """
#     if not msg:
#         return False, False

#     text = msg.lower()

#     nonspicy_keywords_en = [
#         "non spicy",
#         "non-spicy",
#         "no spicy",
#         "without spicy",
#         "without spice",
#         "not spicy",
#         "mild",
#     ]
#     nonspicy_keywords_ar = [
#         "بدون حار",
#         "بدون حر",
#         "عادي",
#     ]

#     nonspicy_flag = any(k in text for k in nonspicy_keywords_en + nonspicy_keywords_ar)

#     spicy_keywords = [
#         "spicy",
#         "hot",
#         "حار",
#     ]
#     spicy_flag = any(k in text for k in spicy_keywords) and not nonspicy_flag

#     return spicy_flag, nonspicy_flag


# def send_category_buttons(user_number: str, lang: str = "en", show_image: bool = True):
#     """
#     Jab user 'Order via text' ya 'Add more items' bole:

#     - show_image=True  -> menu image + category buttons
#     - show_image=False -> sirf category buttons (no image)
#     """

#     caption = "Here is our menu " if lang == "en" else "هذه قائمتنا"

#     if show_image:
#         image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#         try:
#             send_whatsapp_image(user_number, image_url, caption=caption)
#         except Exception as e:
#             print("Error sending menu image in send_category_buttons:", e)

#     if lang == "ar":
#         body = "من فضلك اختر الفئة:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "برجر ووجبات"},
#             {"id": "cat_sandwiches", "title": "ساندويتش"},
#             {"id": "cat_snacks_sides", "title": "وجبات خفيفة"},
#         ]
#     else:
#         body = "Please choose a category:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "Burgers & Meals"},
#             {"id": "cat_sandwiches", "title": "Sandwiches & Wraps"},
#             {"id": "cat_snacks_sides", "title": "Snacks & Sides"},
#         ]

#     send_whatsapp_quick_buttons(user_number, body, buttons)


# def shorten_button_title(title: str, max_len: int = 20) -> str:
#     """
#     WhatsApp Cloud quick-reply button title must be 1–20 chars.
#     If longer, truncate and add '...' so total <= max_len.
#     """
#     title = (title or "").strip()
#     if len(title) <= max_len:
#         return title
#     # 17 chars + "..." = 20
#     return title[: max_len - 3].rstrip() + "..."


# def send_items_for_category(user_number: str, category: str, lang: str = "en"):
#     items = get_items_by_category(category)
#     if not items:
#         send_whatsapp_text(user_number, "No items found in this category yet.")
#         return

#     state = WA_CATEGORY_STATE.get(user_number, {"category": category, "index": 0})
#     index = state.get("index", 0)

#     # if category changed, reset index
#     if state.get("category") != category:
#         index = 0

#     slice_items = items[index : index + 2]
#     buttons = []

#     for item in slice_items:
#         en_name = item["en"]
#         ar_name = item["ar"]

#         if lang == "ar":
#             raw_title = ar_name
#         else:
#             raw_title = en_name.title()

#         title = shorten_button_title(raw_title)  # 🔹 enforce <= 20 chars
#         buttons.append({"id": f"item_{en_name}", "title": title})

#     # "More items" button (already short enough)
#     if index + 2 < len(items):
#         if lang == "ar":
#             more_title = "المزيد من الأصناف"
#         else:
#             more_title = "More items"
#         buttons.append({"id": "more_items", "title": more_title})

#     if lang == "ar":
#         body = "من فضلك اختر الصنف (سيتم إضافة 1 كمية لكل ضغطة):"
#     else:
#         body = "Please choose an item (1 quantity will be added per click):"

#     print("send_items_for_category buttons:", buttons)  # optional debug
#     send_whatsapp_quick_buttons(user_number, body, buttons)

#     WA_CATEGORY_STATE[user_number] = {"category": category, "index": index}


# # ---------------------------
# # LLM Reply
# # ---------------------------


# def get_llm_reply(msg, lang="en"):
#     # If OpenRouter key missing/placeholder, just give a simple fallback message
#     if not OPENROUTER_API_KEY or OPENROUTER_API_KEY.startswith("YOUR_"):
#         return (
#             "You can place your order by telling me the items and quantities."
#             if lang == "en"
#             else "يمكنك وضع طلبك بكتابة الأصناف والكمية."
#         )

#     lang_name = "English" if lang == "en" else "Arabic"

#     sys_prompt = (
#         "You are Joana Fast Food Assistant.\n"
#         "- Short, clear, polite answers.\n"
#         "- Can answer non-restaurant questions briefly, then bring user back to Joana.\n"
#         "- Joana Fast Food is 24/7, takeaway only.\n"
#         f"Always respond in {lang_name}.\n"
#     )

#     context = build_menu_context()

#     messages = [
#         {"role": "system", "content": sys_prompt},
#         {"role": "system", "content": context},
#     ]

#     for m in session.get("messages", []):
#         messages.append(m)

#     messages.append({"role": "user", "content": msg})

#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=messages,
#             temperature=0.5,
#             max_tokens=250,
#         )
#         return res.choices[0].message.content.strip()

#     except Exception as e:
#         print("LLM error:", e)
#         return "Sorry, something went wrong." if lang == "en" else "عذراً، حدث خطأ ما."


# # ---------------------------
# # Routes
# # ---------------------------


# @app.route("/")
# def index():
#     return render_template("index.html")


# @app.route("/cloud-test")
# def cloud_test():
#     # yahan apna khud ka number daal kar test kar sakti ho
#     send_whatsapp_text("9230XXXXXXXX", "Hello from Joana bot via WhatsApp Cloud API!")
#     return "Cloud test message sent!"


# # ---------------------------
# # WhatsApp Cloud Webhook
# # ---------------------------

# @app.route("/whatsapp/webhook", methods=["GET", "POST"])
# def whatsapp_webhook():
#     # 1) Verification handshake (GET)
#     if request.method == "GET":
#         mode = request.args.get("hub.mode")
#         token = request.args.get("hub.verify_token")
#         challenge = request.args.get("hub.challenge")

#         print("Webhook verification:", mode, token, challenge)

#         if mode == "subscribe" and token == WHATSAPP_VERIFY_TOKEN:
#             return challenge or "", 200
#         else:
#             return "Verification token mismatch", 403

#     # 2) Incoming messages (POST)
#     data = request.get_json(force=True) or {}
#     try:
#         print("Incoming from Cloud API:", json.dumps(data, indent=2, ensure_ascii=False))
#     except Exception:
#         print("Incoming from Cloud API (raw):", data)

#     user_text = None
#     user_number = None
#     btn_id = None  # 🔹 yahan id store karenge

#     try:
#         entry = (data.get("entry") or [])[0]
#         changes = (entry.get("changes") or [])[0]
#         value = changes.get("value") or {}
#     except Exception:
#         value = {}

#     # Status callbacks (delivery/read) ignore
#     if value.get("statuses"):
#         print("Cloud API status callback – ignoring")
#         return "ok", 200

#     messages = value.get("messages")
#     if not messages:
#         print("No messages in payload.")
#         return "ignored", 200

#     msg_obj = messages[0]
#     user_number = normalize_number(msg_obj.get("from"))
#     msg_type = msg_obj.get("type")
#     print("Cloud msg_type:", msg_type)

#     if msg_type == "text":
#         user_text = msg_obj.get("text", {}).get("body", "")
#     elif msg_type == "interactive":
#         interactive = msg_obj.get("interactive", {}) or {}
#         itype = interactive.get("type")
#         print("Interactive payload:", interactive)

#         if itype == "button_reply":
#             br = interactive.get("button_reply", {}) or {}
#             btn_id = (br.get("id") or "").strip()
#             user_text = (br.get("title") or btn_id or "").strip()
#             print("Button reply => id:", btn_id, "| title:", user_text)
#         elif itype == "list_reply":
#             lr = interactive.get("list_reply", {}) or {}
#             btn_id = (lr.get("id") or "").strip()
#             user_text = (lr.get("title") or btn_id or "").strip()
#             print("List reply => id:", btn_id, "| title:", user_text)
#     else:
#         # other types (image, audio, etc.) -> simple placeholder
#         user_text = ""

#     if not user_number or not user_text:
#         print("Could not parse user_number or user_text from Cloud payload.")
#         return "ignored", 200

#     print("Parsed RAW:", user_number, "| text:", repr(user_text), "| id:", repr(btn_id))
#     if "\n" in user_text:
#         parts = [p.strip() for p in user_text.split("\n") if p.strip()]
#         if parts:
#             user_text = parts[-1]
#     print("Parsed CLEANED:", user_number, "| text:", repr(user_text))

#     clean = user_text.strip().lower()
#     clean_id = (btn_id or "").strip().lower()   # 🔹 id ka lowercase

#     lang = detect_language(user_text or "")
#     if any("\u0600" <= ch <= "\u06ff" for ch in user_text):
#         lang = "ar"

#     last_state = WHATSAPP_SESSIONS.get(user_number, {}).get("state", {}) or {}
#     last_stage = last_state.get("stage")

#     # 🔥 NEW: handle spicy / non-spicy buttons specially for WhatsApp
#     if last_stage == "await_spice" and clean_id in ["spice_spicy", "spice_non_spicy"]:
#         sess = WHATSAPP_SESSIONS.get(user_number, {})
#         s = sess.get("state") or {}
#         last_item = s.get("last_item")
#         last_qty = s.get("last_qty", 1)

#         if last_item:
#             price, _category = get_price_and_category(last_item)

#             # Ensure order list exists
#             order_list = s.get("order")
#             if not isinstance(order_list, list):
#                 order_list = []
#             s["order"] = order_list

#             if clean_id == "spice_spicy":
#                 spicy_q = last_qty
#                 non_q = 0
#             else:  # spice_non_spicy
#                 spicy_q = 0
#                 non_q = last_qty

#             if spicy_q > 0:
#                 order_list.append(
#                     {
#                         "item": last_item,
#                         "qty": spicy_q,
#                         "spicy": 1,
#                         "nonspicy": 0,
#                         "price": price,
#                         "subtotal": spicy_q * price,
#                     }
#                 )

#             if non_q > 0:
#                 order_list.append(
#                     {
#                         "item": last_item,
#                         "qty": non_q,
#                         "spicy": 0,
#                         "nonspicy": 1,
#                         "price": price,
#                         "subtotal": non_q * price,
#                     }
#                 )

#             # reset stage & last item, go to add_more
#             s["stage"] = "add_more"
#             s["last_item"] = None
#             s["last_qty"] = 0
#             sess["state"] = s
#             WHATSAPP_SESSIONS[user_number] = sess

#             # reply text
#             if lang == "ar":
#                 kind = "حار" if clean_id == "spice_spicy" else "بدون حار"
#                 reply_text = (
#                     f"تمت إضافة {last_item} {kind} ×{last_qty}! "
#                     "هل ترغب في إضافة شيء آخر؟"
#                 )
#                 buttons = [
#                     {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
#                     {"id": "finish_order", "title": "إنهاء الطلب"},
#                 ]
#             else:
#                 kind = "spicy" if clean_id == "spice_spicy" else "non-spicy"
#                 reply_text = (
#                     f"Added {last_qty} {kind} {last_item.title()}! "
#                     "Would you like to add anything else?"
#                 )
#                 buttons = [
#                     {"id": "add_more_items", "title": "Add more items"},
#                     {"id": "finish_order", "title": "Finish order"},
#                 ]

#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200
#         # if for some reason last_item missing, fall through to normal logic

#     # 1️⃣ FIRST MESSAGE on WhatsApp -> always show branches + "Order via text / voice"
#     if last_stage is None:
#         if lang == "ar":
#             buttons = [
#                 {"id": "order_text", "title": "الطلب عبر الرسائل"},
#                 {"id": "order_voice", "title": "الطلب عبر الصوت"},
#             ]
#             body = build_branches_message("ar")
#         else:
#             buttons = [
#                 {"id": "order_text", "title": "Order via text"},
#                 {"id": "order_voice", "title": "Order via voice"},
#             ]
#             body = build_branches_message("en")

#         send_whatsapp_quick_buttons(user_number, body, buttons)
#         # initialize empty session
#         WHATSAPP_SESSIONS[user_number] = {"state": {"stage": "branch"}, "messages": []}
#         return "ok", 200

#     # 2️⃣ Category selection (title + id dono se match)
#     if (
#         clean in ["burgers & meals", "burgers_meals", "برجر ووجبات"]
#         or clean_id == "cat_burgers_meals"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "burgers_meals", "index": 0}
#         send_items_for_category(user_number, "burgers_meals", lang)
#         return "ok", 200

#     if (
#         clean in ["sandwiches & wraps", "sandwiches", "ساندويتش", "ساندويتشات"]
#         or clean_id == "cat_sandwiches"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "sandwiches", "index": 0}
#         send_items_for_category(user_number, "sandwiches", lang)
#         return "ok", 200

#     if (
#         clean in ["snacks & sides", "snacks_sides", "وجبات خفيفة"]
#         or clean_id == "cat_snacks_sides"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "snacks_sides", "index": 0}
#         send_items_for_category(user_number, "snacks_sides", lang)
#         return "ok", 200

#     # 3️⃣ More items in category (English + Arabic both)
#     if (
#         clean in ["more items", "المزيد من الأصناف"]           # exact button title
#         or ("more" in clean and "item" in clean)                # user manually typed
#         or clean_id == "more_items"                             # button id
#     ):
#         print("➡ MORE ITEMS clicked | clean =", clean, "| id =", clean_id)

#         st = WA_CATEGORY_STATE.get(user_number, {})
#         cat = st.get("category")
#         idx = st.get("index", 0)

#         if not cat:
#             print("⚠ No category saved for this user in WA_CATEGORY_STATE")
#         else:
#             items = get_items_by_category(cat)
#             if items:
#                 new_index = idx + 2
#                 if new_index >= len(items):
#                     # wrap to start
#                     new_index = 0

#                 WA_CATEGORY_STATE[user_number] = {
#                     "category": cat,
#                     "index": new_index,
#                 }

#                 print(
#                     f"➡ Paging items for category={cat}, "
#                     f"old_index={idx}, new_index={new_index}, total_items={len(items)}"
#                 )

#                 send_items_for_category(user_number, cat, lang)
#                 return "ok", 200

#     # 4️⃣ Add more / finish order buttons (from interactive)
#     if (
#         clean in ["add more items", "إضافة أصناف أخرى", "اضافة اصناف اخرى"]
#         or clean_id in ["add_more_items", "payment_add_more"]
#     ) and last_stage in ["add_more", "payment"]:
#         sess = WHATSAPP_SESSIONS.get(user_number)
#         if isinstance(sess, dict):
#             st = sess.get("state") or {}
#             st["stage"] = "add_more"
#             sess["state"] = st
#             WHATSAPP_SESSIONS[user_number] = sess

#         send_category_buttons(user_number, lang, show_image=False)
#         return "ok", 200

#     if clean in ["finish order", "إنهاء الطلب", "انهاء الطلب"] or clean_id == "finish_order":
#         user_text = "done"
#         clean = "done"

#     # 5️⃣ "Order via text" button
#     if clean_id == "order_text" or clean in [
#         "1",
#         "1.",
#         "order via text",
#         "order by text",
#         "text order",
#         "الطلب عبر الرسائل",
#         "الطلب بالرسائل",
#         "اطلب بالرسائل",
#     ]:
#         send_category_buttons(user_number, lang, show_image=True)
#         return "ok", 200

#     # 6️⃣ "Order via voice"
#     if clean_id == "order_voice" or clean in [
#         "2",
#         "2.",
#         "order via voice",
#         "order by voice",
#         "voice order",
#         "voice",
#         "الطلب عبر الصوت",
#         "الطلب بالصوت",
#         "اطلب بالصوت",
#     ]:
#         send_whatsapp_text(
#             user_number,
#             "🎙 من فضلك أرسل طلبك على شكل رسالة صوتية.\n\n"
#             'مثال: "اثنين زنجر برجر، واحد حار وواحد بدون حار، وبطاطس كبيرة".\n\n'
#             "سيقوم الموظف بالاستماع وتأكيد طلبك 😊",
#         )
#         return "ok", 200

#     # 7️⃣ Baaki sab normal chatbot flow (aap ka purana code yahan se same)
#     try:
#         result = process_whatsapp_message(user_number, user_text, lang)

#         reply_html = result.get("reply") or "Sorry, something went wrong."
#         reply_text = html_to_whatsapp(reply_html)

#         stage = result.get("stage")
#         order = result.get("order") or []

#         if result.get("menu"):
#             image_url = (
#                 "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#             )
#             send_whatsapp_image(user_number, image_url, caption="Here is our menu 📋")

#         # Spicy selection buttons
#         if stage == "await_spice":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "spice_spicy", "title": "حار"},
#                     {"id": "spice_non_spicy", "title": "بدون حار"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "spice_spicy", "title": "Spicy"},
#                     {"id": "spice_non_spicy", "title": "Non-spicy"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Add more / finish order buttons (after adding an item)
#         if stage == "add_more" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
#                     {"id": "finish_order", "title": "إنهاء الطلب"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "add_more_items", "title": "Add more items"},
#                     {"id": "finish_order", "title": "Finish order"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Payment page buttons
#         if stage == "payment" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "payment_proceed", "title": "المتابعة إلى الدفع"},
#                     {"id": "payment_add_more", "title": "إضافة أصناف أخرى"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "payment_proceed", "title": "Proceed to payment"},
#                     {"id": "payment_add_more", "title": "Add more items"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Choose payment method buttons
#         if stage == "choose_payment":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "pay_online", "title": "الدفع عبر الإنترنت"},
#                     {"id": "pay_cash", "title": "الدفع نقداً"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "pay_online", "title": "Online payment"},
#                     {"id": "pay_cash", "title": "Cash"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#     except Exception as e:
#         print("Error in whatsapp_webhook:", e)
#         reply_text = "Sorry, something went wrong."

#     send_whatsapp_text(user_number, reply_text)
#     return "ok", 200


# def process_whatsapp_message(
#     user_number: str, user_text: str, lang_hint: str = None
# ) -> dict:
#     ctx = WHATSAPP_SESSIONS.get(user_number, {})
#     prev_state = ctx.get("state")
#     prev_messages = ctx.get("messages")

#     with app.test_request_context(
#         "/api/chat",
#         method="POST",
#         json={
#             "message": user_text,
#             "is_voice": False,
#             "lang_hint": lang_hint,
#         },
#     ):
#         if prev_state is not None:
#             session["state"] = prev_state
#         if prev_messages is not None:
#             session["messages"] = prev_messages

#         resp = chat()

#         WHATSAPP_SESSIONS[user_number] = {
#             "state": session.get("state"),
#             "messages": session.get("messages"),
#         }

#         if isinstance(resp, tuple):
#             resp_obj = resp[0]
#         else:
#             resp_obj = resp

#         data = resp_obj.get_json() or {}
#         return data


# # ---------------------------
# # Web chat route
# # ---------------------------


# def make_chat_response(reply, lang, menu=None):
#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     payload = {
#         "reply": reply,
#         "lang": lang,
#         "stage": s.get("stage"),
#         "order": s.get("order"),
#         "total": s.get("total", 0),
#     }
#     if menu:
#         payload["menu"] = menu
#     return jsonify(payload)


# @app.route("/api/chat", methods=["POST"])
# def chat():
#     global MENU, BRANCHES
#     MENU = load_menu()
#     BRANCHES = load_branches()

#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     session["state"] = s
#     session["messages"] = session.get("messages", [])

#     data = request.get_json(force=True)
#     msg_raw = (data.get("message") or "").strip()
#     msg = msg_raw
#     is_voice = data.get("is_voice", False)
#     lang_hint = data.get("lang_hint")

#     lang = detect_language(msg_raw)

#     if any("\u0600" <= ch <= "\u06ff" for ch in msg_raw):
#         lang = "ar"

#     if lang_hint in ("en", "ar"):
#         lang = lang_hint

#     if lang == "ar":
#         msg = llm_normalize_text(msg_raw, lang)

#     intent = detect_intent(msg)

#     session["messages"].append({"role": "user", "content": msg})

#     msg_l = msg.lower()
#     if "menu" in msg_l or "send menu" in msg_l or "price list" in msg_l:
#         intent = "menu"

#     if polite_check(msg):
#         reply = "Please speak politely." if lang == "en" else "من فضلك تحدث بأدب."
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     AR_GREETINGS = [
#         "مرحبا",
#         "مرحبا بك",
#         "اهلا",
#         "أهلا",
#         "اهلاً",
#         "اهلا وسهلا",
#         "السلام عليكم",
#         "السلام",
#         "هلا",
#         "يا هلا",
#         "مساء الخير",
#         "صباح الخير",
#     ]

#     EN_GREETINGS = [
#         "hi",
#         "hii",   # NEW
#         "hello",
#         "hey",
#         "good morning",
#         "good evening",
#         "good afternoon",
#         "hiya",
#         "yo",
#     ]

#     def is_greeting_text(msg_: str, lang_: str) -> bool:
#         """
#         Sirf tab greeting ho jab pure text me sirf greeting hi ho.
#         'chicken sandwich' jese text ko greeting na samjho.
#         """
#         if not msg_:
#             return False

#         text = msg_.strip().lower()
#         text = re.sub(r"[^\w\u0600-\u06FF ]+", "", text)

#         greetings = AR_GREETINGS if lang_ == "ar" else EN_GREETINGS
#         return text in greetings

#     # Greeting (only if stage empty) – for web chat
#     if s.get("stage") is None and is_greeting_text(msg, lang):
#         s = {
#             "stage": "branch",
#             "order": [],
#             "total": 0,
#             "last_item": None,
#             "last_qty": 0,
#         }
#         session["state"] = s

#         if lang == "ar":
#             reply = (
#                 "مرحباً بك في مطعم جوانا للوجبات السريعة! 🍔<br>"
#                 "ابحث عن أقرب فرع واتصل على الرقم للطلب:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>جاهز للطلب؟ شارك طلبك بالصوت أو بالكتابة!<br><br>"
#             )
#         else:
#             reply = (
#                 "Welcome to JOANA Fast Food! <br>"
#                 "Find your nearest branch and dial the number to order:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>Ready to order? Share your order via voice or chat!<br><br>"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     if intent == "menu":
#         reply = (
#             "Here’s our menu! Please place your order."
#             if lang == "en"
#             else "هذه قائمتنا! من فضلك ضع طلبك."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang, menu="/static/menu.PNG")

#     # user says "add more" in payment (free text)
#     if s.get("stage") == "payment" and (
#         "add more" in msg.lower() or "إضافة" in msg or "اضافة" in msg or "أخرى" in msg
#     ):
#         s["stage"] = "add_more"
#         session["state"] = s
#         reply = (
#             "Sure! Please tell me what else you would like to add to your order."
#             if lang == "en"
#             else "حسناً! من فضلك أخبرني ما الذي تريد إضافته أيضاً."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     # Spicy follow-up
#     if s.get("stage") == "await_spice" and s.get("last_item"):
#         last_item = s["last_item"]
#         last_qty = s.get("last_qty", 1)
#         price, _category = get_price_and_category(last_item)
#         split = parse_spice_split(msg)

#         if split:
#             spicy_q, non_q = split
#         else:
#             spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#             if spicy_flag:
#                 spicy_q, non_q = last_qty, 0
#             elif nonspicy_flag:
#                 spicy_q, non_q = 0, last_qty
#             else:
#                 spicy_q, non_q = 0, last_qty

#         order_added_msgs = []

#         if spicy_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": spicy_q,
#                     "spicy": 1,
#                     "nonspicy": 0,
#                     "price": price,
#                     "subtotal": spicy_q * price,
#                 }
#             )
#             order_added_msgs.append(f"{spicy_q} {'spicy' if lang == 'en' else 'حار'}")

#         if non_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": non_q,
#                     "spicy": 0,
#                     "nonspicy": 1,
#                     "price": price,
#                     "subtotal": non_q * price,
#                 }
#             )
#             order_added_msgs.append(
#                 f"{non_q} {'non-spicy' if lang == 'en' else 'بدون حار'}"
#             )

#         s["stage"] = "add_more"
#         s["last_item"] = None
#         s["last_qty"] = 0
#         session["state"] = s

#         reply = (
#             f"Added {last_item.title()} — "
#             + " & ".join(order_added_msgs)
#             + ". Would you like to add anything else?"
#             if lang == "en"
#             else f"تمت إضافة {last_item} — "
#             + " و ".join(order_added_msgs)
#             + ". هل ترغب في إضافة شيء آخر؟"
#         )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Item detection
#     found = find_menu_item(msg)

#     if found:
#         qty = detect_qty(msg)
#         price, category = get_price_and_category(found)

#         s["last_item"] = found
#         s["last_qty"] = qty
#         session["state"] = s

#         spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#         if category == "burgers_meals" and not (spicy_flag or nonspicy_flag):
#             s["stage"] = "await_spice"
#             session["state"] = s

#             reply = (
#                 f"Would you like your {found.title()} spicy or non-spicy?"
#                 if lang == "en"
#                 else f"هل ترغب أن يكون {found} حارًا أم بدون حار؟"
#             )

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         if category == "burgers_meals":
#             if spicy_flag:
#                 spicy = 1
#                 nonspicy = 0
#             elif nonspicy_flag:
#                 spicy = 0
#                 nonspicy = 1
#             else:
#                 spicy = 0
#                 nonspicy = 1
#         else:
#             spicy = 0
#             nonspicy = 0

#         s["order"].append(
#             {
#                 "item": found,
#                 "qty": qty,
#                 "spicy": spicy,
#                 "nonspicy": nonspicy,
#                 "price": price,
#                 "subtotal": qty * price,
#             }
#         )

#         previous_stage = s.get("stage")
#         s["last_item"] = None
#         s["last_qty"] = 0

#         if previous_stage == "payment":
#             summary, total = build_order_summary_and_total(s["order"], lang)
#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             reply = (
#                 "Your order has been updated! Here's the new summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تحديث طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )
#         else:
#             s["stage"] = "add_more"
#             session["state"] = s

#             reply = (
#                 f"{found.title()} ×{qty} added! Would you like to add anything else?"
#                 if lang == "en"
#                 else f"تمت إضافة {found} ×{qty}! هل ترغب في إضافة شيء آخر؟"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Done / no more items
#     if any(x in msg.lower() for x in ["no", "done", "that's all", "that all"]) or any(
#         x in msg for x in ["خلص", "لا", "انتهى", "كفاية"]
#     ):
#         if s.get("order"):

#             summary, total = build_order_summary_and_total(s["order"], lang)

#             reply = (
#                 "Your order is confirmed! Here's a summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تأكيد طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Payment yes
#     if s.get("stage") == "payment" and (
#         any(
#             x in msg.lower()
#             for x in [
#                 "yes",
#                 "sure",
#                 "ok",
#                 "okay",
#                 "proceed",
#                 "go ahead",
#             ]
#         )
#         or any(
#             x in msg
#             for x in [
#                 "تمام",
#                 "نعم",
#                 "أكيد",
#                 "المتابعة",
#                 "تابع",
#                 "ادفع",
#                 "إلى الدفع",
#             ]
#         )
#     ):
#         reply = (
#             "How would you like to pay?<br><b>Cash</b> or <b>Online Payment</b>?"
#             if lang == "en"
#             else "كيف ترغب في الدفع؟<br><b> نقدي </b> أو <b> الدفع عبر الإنترنت</b>؟"
#         )
#         s["stage"] = "choose_payment"
#         session["state"] = s

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Choose payment
#     if s.get("stage") == "choose_payment":
#         total = s.get("total", 0)
#         msg_l = msg.lower()

#         if any(x in msg_l for x in ["cash", "cod", "نقد", "نقدا"]):
#             reply = (
#                 "Thank you for your order! Your order will be ready in 20 to 30 minutes."
#                 if lang == "en"
#                 else "شكراً لطلبك! سيكون جاهزاً خلال 20 إلى 30 دقيقة."
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         elif any(
#             x in msg_l
#             for x in [
#                 "online",
#                 "online payment",
#                 "pay online",
#                 "card",
#                 "visa",
#                 "master",
#                 "debit",
#                 "mada",
#             ]
#         ) or any(x in msg for x in ["مدى", "إلكتروني", "دفع", "كرت"]):
#             reply = (
#                 f"Great! Your payment of {total:.2f} {CURRENCY} is being processed.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>Click here to complete your payment</b></a>"
#                 if lang == "en"
#                 else f"رائع! يتم الآن معالجة دفعتك بمقدار {total:.2f} {CURRENCY_AR}.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>اضغط هنا لإتمام الدفع</b></a>"
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         else:
#             reply = (
#                 "Please select a valid payment method — Cash or Online."
#                 if lang == "en"
#                 else "يرجى اختيار طريقة الدفع الصحيحة - نقدًا أو عبر الإنترنت"
#             )
#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Generic LLM reply
#     reply = get_llm_reply(msg, lang)

#     if s.get("stage") == "payment" and s.get("order"):
#         summary, total = build_order_summary_and_total(s["order"], lang)
#         s["total"] = total
#         session["state"] = s

#         if lang == "en":
#             reply = (
#                 reply
#                 + "<br><br>Here is your current order summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#             )
#         else:
#             reply = (
#                 reply
#                 + "<br><br>هذا ملخص طلبك الحالي:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#     if is_voice:
#         speak_text(reply, lang)

#     return make_chat_response(reply, lang)


# # ---------------------------
# # Run App (local)
# # ---------------------------
# if __name__ == "__main__":
#     app.run(debug=True)


# Englis and Arabic working fine, only the more items button bug
# import json
# import os
# import re
# import sqlite3
# import requests

# import pandas as pd
# from flask import Flask, render_template, request, jsonify, session
# from nlp_utils import detect_intent, detect_language
# from openai import OpenAI
# from dotenv import load_dotenv

# load_dotenv()

# # ---------------------------
# # ENV VARIABLES
# # ---------------------------
# OPENROUTER_API_KEY = os.getenv("OPENROUTER_API_KEY", "YOUR_OPENROUTER_API_KEY_HERE")
# FLASK_SECRET = os.getenv("FLASK_SECRET_KEY", "joana-fastfood-secret")

# # WhatsApp Cloud API envs
# WHATSAPP_TOKEN = os.getenv("WHATSAPP_TOKEN")
# WHATSAPP_PHONE_NUMBER_ID = os.getenv("WHATSAPP_PHONE_NUMBER_ID")
# WHATSAPP_VERIFY_TOKEN = os.getenv("WHATSAPP_VERIFY_TOKEN", "joana-verify-token-123")

# WHATSAPP_API_BASE = "https://graph.facebook.com/v21.0"

# # ---------------------------
# # WHATSAPP CLOUD SEND HELPERS
# # ---------------------------


# def send_whatsapp_text(to_number: str, text: str):
#     """
#     Send plain text via WhatsApp Cloud API
#     """
#     print("=== send_whatsapp_text CALLED ===")
#     print("WHATSAPP_TOKEN present?", bool(WHATSAPP_TOKEN))
#     print("Sending to:", to_number, "| Text:", text)

#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "text",
#         "text": {"body": text},
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD SEND RESPONSE:", resp.status_code, resp.text)
#     except Exception as e:
#         print("Error sending WhatsApp text via Cloud API:", repr(e))


# def send_whatsapp_image(to_number: str, image_url: str, caption: str = ""):
#     """
#     Send image + optional caption via WhatsApp Cloud API
#     """
#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "image",
#         "image": {
#             "link": image_url,
#         },
#     }
#     if caption:
#         payload["image"]["caption"] = caption

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD IMAGE SEND:", resp.status_code, resp.text)
#         return resp.status_code in (200, 201)
#     except Exception as e:
#         print("Error sending WhatsApp image via Cloud API:", e)
#         return False


# def send_whatsapp_quick_buttons(to_number: str, body_text: str, buttons: list):
#     """
#     Send interactive "button" template via WhatsApp Cloud API
#     (max 3 buttons)
#     """
#     if not (WHATSAPP_TOKEN and WHATSAPP_PHONE_NUMBER_ID):
#         print("Missing WHATSAPP_TOKEN or WHATSAPP_PHONE_NUMBER_ID")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }

#     # Cloud API button format
#     wa_buttons = []
#     for b in buttons[:3]:
#         wa_buttons.append(
#             {
#                 "type": "reply",
#                 "reply": {
#                     "id": b.get("id", b["title"]),
#                     "title": b["title"],
#                 },
#             }
#         )

#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "interactive",
#         "interactive": {
#             "type": "button",
#             "body": {"text": body_text},
#             "action": {"buttons": wa_buttons},
#         },
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD BUTTONS SEND:", resp.status_code, resp.text)
#         return resp.status_code in (200, 201)
#     except Exception as e:
#         print("Error sending WhatsApp buttons via Cloud API:", e)
#         return False


# # ---------------------------
# # Paths / Directories
# # ---------------------------
# BASE_DIR = os.path.dirname(os.path.abspath(__file__))
# DATA_DIR = os.path.join(BASE_DIR, "data")

# MENU_FILE = os.path.join(DATA_DIR, "Menu.xlsx")
# BRANCHES_FILE = os.path.join(DATA_DIR, "Branches.xlsx")

# WHATSAPP_SESSIONS = {}
# WA_CATEGORY_STATE = {}  # phone -> {"category": str, "index": int}

# # ---------------------------
# # Flask Config
# # ---------------------------
# app = Flask(__name__, static_folder="static", template_folder="templates")
# app.secret_key = FLASK_SECRET

# client = OpenAI(api_key=OPENROUTER_API_KEY, base_url="https://openrouter.ai/api/v1")

# PAYMENT_URL = (
#     "https://starlit-sopapillas-520aa2.netlify.app/"
#     "?redirect=http://127.0.0.1:5000/thankyou"
# )
# CURRENCY = "SAR"
# CURRENCY_AR = "ريال سعودي"

# # ---------------------------
# # Database
# # ---------------------------
# DB_PATH = os.path.join(DATA_DIR, "orders.db")


# def init_db():
#     os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
#     conn = sqlite3.connect(DB_PATH)
#     conn.execute(
#         """
#         CREATE TABLE IF NOT EXISTS orders (
#             id INTEGER PRIMARY KEY AUTOINCREMENT,
#             item TEXT,
#             qty INTEGER,
#             spicy INTEGER,
#             nonspicy INTEGER,
#             price REAL,
#             subtotal REAL,
#             total REAL,
#             payment_method TEXT,
#             datetime TEXT
#         )
#         """
#     )
#     conn.commit()
#     conn.close()


# init_db()

# # ---------------------------
# # Category mapping helper
# # ---------------------------


# def canonical_category(raw_category: str, name_en: str = "") -> str:
#     """
#     Excel category ko 3 main groups me convert:
#     burgers_meals, sandwiches, snacks_sides
#     """
#     c = (raw_category or "").strip().lower()
#     n = (name_en or "").strip().lower()

#     if c in ["burgers", "burger", "meals", "meal"]:
#         return "burgers_meals"
#     if c == "sandwiches":
#         return "sandwiches"
#     if c in ["sides", "side", "drinks", "drink"]:
#         return "snacks_sides"

#     if any(
#         k in n for k in ["burger", "zinger", "tortilla", "hot dog", "hotdog", "meal"]
#     ):
#         return "burgers_meals"
#     if any(k in n for k in ["sandwich", "kudu", "kabab"]):
#         return "sandwiches"
#     if any(
#         k in n
#         for k in [
#             "fries",
#             "nuggets",
#             "rings",
#             "juice",
#             "corn",
#             "potato",
#             "samosa",
#             "popcorn",
#         ]
#     ):
#         return "snacks_sides"

#     return ""


# # ---------------------------
# # Excel Loaders
# # ---------------------------


# def load_menu():
#     try:
#         df_raw = pd.read_excel(MENU_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             if any("name_en" in str(c).lower() for c in row):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             print("❌ Could not find header row containing 'name_en'")
#             return {}

#         df = pd.read_excel(MENU_FILE, header=header_row_index)

#         def has(col, *keys):
#             c = str(col).strip().lower()
#             return any(k in c for k in keys)

#         english_col = next((c for c in df.columns if has(c, "name_en")), None)
#         arabic_col = next((c for c in df.columns if has(c, "name_ar")), None)
#         price_col = next((c for c in df.columns if has(c, "price")), None)
#         cat_col = next(
#             (c for c in df.columns if has(c, "category", "cat", "type")), None
#         )

#         if not english_col or not price_col:
#             raise Exception("Missing name_en or price column")

#         menu = {}
#         for _, row in df.iterrows():
#             en = str(row[english_col]).strip().lower()
#             if not en or en == "nan":
#                 continue

#             ar = str(row[arabic_col]).strip() if arabic_col else en
#             raw_cat = str(row.get(cat_col, "")).strip() if cat_col else ""

#             try:
#                 price = float(row[price_col])
#             except Exception:
#                 continue

#             cat_final = canonical_category(raw_cat, en)

#             entry = {
#                 "price": price,
#                 "category": cat_final,
#                 "name_en": en,
#                 "name_ar": ar or en,
#             }

#             menu[en] = entry
#             if ar:
#                 menu[ar.lower()] = entry

#         print(f"✅ Loaded {len(menu)} menu items.")
#         return menu

#     except Exception as e:
#         print("Menu load failed:", e)
#         return {}


# def load_branches():
#     try:
#         df_raw = pd.read_excel(BRANCHES_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             row_l = [str(c).lower() for c in row]
#             if any("branch" in c for c in row_l) and any("address" in c for c in row_l):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             header_row_index = 0

#         df = pd.read_excel(BRANCHES_FILE, header=header_row_index)

#         name_col = next((c for c in df.columns if "branch" in str(c).lower()), None)
#         addr_col = next((c for c in df.columns if "address" in str(c).lower()), None)
#         phone_col = next(
#             (
#                 c
#                 for c in df.columns
#                 if "phone" in str(c).lower() or "number" in str(c).lower()
#             ),
#             None,
#         )

#         branches = []
#         for _, row in df.iterrows():
#             branches.append(
#                 {
#                     "Branch Name": str(row.get(name_col, "")).strip(),
#                     "Address / Area": str(row.get(addr_col, "")).strip(),
#                     "Phone Number": str(row.get(phone_col, "")).strip(),
#                 }
#             )

#         print(f"✅ Loaded {len(branches)} branches.")
#         return [
#             b
#             for b in branches
#             if (b["Branch Name"] or b["Address / Area"] or b["Phone Number"])
#         ]

#     except Exception as e:
#         print("❌ Branch load failed:", e)
#         return []


# MENU = load_menu()
# BRANCHES = load_branches()

# # ---------------------------
# # Helpers
# # ---------------------------


# def build_menu_context():
#     if not MENU:
#         return "Current restaurant menu is empty."

#     lines = []
#     for name, info in MENU.items():
#         if not re.search(r"[A-Za-z]", name):
#             continue
#         price = info.get("price", 0.0)
#         category = info.get("category", "") or ""
#         lines.append(f"- {name} | price: {price:.2f} {CURRENCY} | category: {category}")

#     if not lines:
#         return "Current restaurant menu is empty."

#     return "Current restaurant menu (items, prices, categories):\n" + "\n".join(lines)


# def build_branches_message(lang: str = "en") -> str:
#     lines = []
#     for b in BRANCHES[:6]:
#         name = b.get("Branch Name", "")
#         addr = b.get("Address / Area", "")
#         phone = b.get("Phone Number", "")
#         if phone:
#             lines.append(f"- {name} – {addr} | {phone}")

#     branches_text = "\n".join(lines) if lines else "Branches info not available."

#     if lang == "ar":
#         return (
#             "مرحباً بك في مطعم جوانا للوجبات السريعة!\n\n"
#             "الفروع:\n"
#             f"{branches_text}\n\n"
#             "من فضلك اختر طريقة الطلب:"
#         )
#     else:
#         return (
#             "Welcome to JOANA Fast Food! \n\n"
#             "Branches:\n"
#             f"{branches_text}\n\n"
#             "Please choose an option:"
#         )


# def html_to_whatsapp(text: str) -> str:
#     if not text:
#         return ""

#     text = text.replace("<br>", "\n").replace("<br/>", "\n").replace("<br />", "\n")

#     def replace_anchor(match):
#         full_tag = match.group(0)
#         label = match.group(1).strip()

#         href_match = re.search(
#             r'href=[\'"]([^\'"]+)[\'"]', full_tag, flags=re.IGNORECASE
#         )
#         if not href_match:
#             return label

#         url = href_match.group(1).strip()

#         if url.startswith("tel:"):
#             return label

#         return f"{label}: {url}"

#     text = re.sub(
#         r"<a[^>]>(.?)</a>", replace_anchor, text, flags=re.IGNORECASE | re.DOTALL
#     )
#     text = re.sub(r"<[^>]+>", "", text)
#     return text.strip()


# def normalize_number(raw: str) -> str:
#     if not raw:
#         return ""
#     raw = str(raw)
#     if raw.startswith("+"):
#         raw = raw[1:]
#     return raw


# WA_GREETINGS = [
#     "hi",
#     "hello",
#     "hey",
#     "salam",
#     "slam",
#     "asalam",
#     "assalam",
#     "assalam o alaikum",
#     "assalamu alaikum",
#     "السلام عليكم",
#     "مرحبا",
# ]


# def is_wa_greeting(text: str) -> bool:
#     """
#     WhatsApp greeting sirf tab jab pura text hi greeting ho.
#     'chicken sandwich' mein 'hi' aata hai lekin yeh greeting nahi hai.
#     """
#     if not text:
#         return False
#     t = text.strip().lower()
#     t = re.sub(r"[^\w\u0600-\u06FF ]+", "", t)
#     return t in WA_GREETINGS


# def llm_normalize_text(text: str, lang: str) -> str:
#     if not text or lang != "ar":
#         return text
#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=[
#                 {
#                     "role": "system",
#                     "content": (
#                         "You are a spell-correction engine for user messages.\n"
#                         "- Fix Arabic spelling, don't translate.\n"
#                         "- Return only corrected text."
#                     ),
#                 },
#                 {"role": "user", "content": text},
#             ],
#             temperature=0,
#             max_tokens=100,
#         )
#         return res.choices[0].message.content.strip()
#     except Exception as e:
#         print("LLM normalize error:", e)
#         return text


# def detect_qty(msg: str) -> int:
#     if not msg:
#         return 1
#     text = msg.lower().strip()

#     arabic_digit_map = {
#         "٠": "0",
#         "١": "1",
#         "٢": "2",
#         "٣": "3",
#         "٤": "4",
#         "٥": "5",
#         "٦": "6",
#         "٧": "7",
#         "٨": "8",
#         "٩": "9",
#     }
#     for ar, en in arabic_digit_map.items():
#         text = text.replace(ar, en)

#     digit_match = re.search(r"\b(\d+)\b", text)
#     if digit_match:
#         try:
#             q = int(digit_match.group(1))
#             if q > 0:
#                 return q
#         except ValueError:
#             pass

#     number_words_en = {
#         "one": 1,
#         "two": 2,
#         "three": 3,
#         "four": 4,
#         "five": 5,
#         "six": 6,
#         "seven": 7,
#         "eight": 8,
#         "nine": 9,
#         "ten": 10,
#     }
#     for word, value in number_words_en.items():
#         if re.search(r"\b" + re.escape(word) + r"\b", text):
#             return value

#     number_words_ar = {
#         "واحد": 1,
#         "واحدة": 1,
#         "اثنين": 2,
#         "اتنين": 2,
#         "ثنين": 2,
#         "ثلاثة": 3,
#         "ثلاث": 3,
#         "اربعة": 4,
#         "أربعة": 4,
#         "خمسة": 5,
#         "ستة": 6,
#         "سبعة": 7,
#         "ثمانية": 8,
#         "تسعة": 9,
#         "عشرة": 10,
#     }
#     for word, value in number_words_ar.items():
#         if word in text:
#             return value

#     return 1


# def polite_check(text):
#     bad = ["idiot", "stupid", "حرام", "لعنة", "غبي"]
#     return any(w in text.lower() for w in bad)


# def get_price_and_category(name):
#     entry = MENU.get(name.lower(), {})
#     return float(entry.get("price", 0)), entry.get("category", "")


# def build_order_summary_and_total(order_items, lang):
#     total = sum(i.get("subtotal", 0) for i in order_items)
#     summary = []

#     for i in order_items:
#         item_name = i["item"]
#         item_info = MENU.get(item_name.lower(), {})
#         category = item_info.get("category")

#         if category == "burgers_meals":
#             if lang == "ar":
#                 kind = "حار" if i.get("spicy") else "بدون حار"
#             else:
#                 kind = "spicy" if i.get("spicy") else "non-spicy"
#             summary.append(f"{i['qty']} {kind} {item_name.title()}")
#         else:
#             summary.append(f"{i['qty']} {item_name.title()}")

#     return summary, total


# def speak_text(reply, lang):
#     return


# def parse_spice_split(msg):
#     text = msg.lower().replace("-", " ")
#     nums = re.findall(r"\d+", text)

#     if len(nums) >= 2:
#         n1, n2 = int(nums[0]), int(nums[1])

#         if ("spicy" in text or "حار" in text) and (
#             "non" in text or "mild" in text or "عادي" in text or "بدون" in text
#         ):
#             spicy_first = (
#                 text.find("spicy") < text.find("non") if "non" in text else True
#             )
#             arab_spicy_first = (
#                 text.find("حار") < text.find("عادي") if "عادي" in text else True
#             )

#             return (n1, n2) if (spicy_first or arab_spicy_first) else (n2, n1)

#     return None


# def find_menu_item(msg):
#     text = msg.lower()
#     candidates = []

#     for name in MENU.keys():
#         pattern = r"\b" + re.escape(name) + r"s?\b"
#         if re.search(pattern, text):
#             candidates.append(name)

#     if not candidates:
#         return None

#     candidates.sort(key=len, reverse=True)
#     return candidates[0]


# def get_items_by_category(category: str):
#     """
#     Return unique items for a category:
#     [{"en": "...", "ar": "..."}]
#     """
#     items_map = {}
#     for key, info in MENU.items():
#         if info.get("category") != category:
#             continue
#         en = (info.get("name_en") or key).strip().lower()
#         ar = (info.get("name_ar") or en).strip()
#         items_map[en] = {"en": en, "ar": ar}
#     return list(items_map.values())


# def detect_spicy_nonspicy(msg: str):
#     """
#     Text se spicy / non-spicy flags nikalta hai.
#     """
#     if not msg:
#         return False, False

#     text = msg.lower()

#     nonspicy_keywords_en = [
#         "non spicy",
#         "non-spicy",
#         "no spicy",
#         "without spicy",
#         "without spice",
#         "not spicy",
#         "mild",
#     ]
#     nonspicy_keywords_ar = [
#         "بدون حار",
#         "بدون حر",
#         "عادي",
#     ]

#     nonspicy_flag = any(k in text for k in nonspicy_keywords_en + nonspicy_keywords_ar)

#     spicy_keywords = [
#         "spicy",
#         "hot",
#         "حار",
#     ]
#     spicy_flag = any(k in text for k in spicy_keywords) and not nonspicy_flag

#     return spicy_flag, nonspicy_flag


# def send_category_buttons(user_number: str, lang: str = "en", show_image: bool = True):
#     """
#     Jab user 'Order via text' ya 'Add more items' bole:

#     - show_image=True  -> menu image + category buttons
#     - show_image=False -> sirf category buttons (no image)
#     """

#     caption = "Here is our menu " if lang == "en" else "هذه قائمتنا"

#     if show_image:
#         image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#         try:
#             send_whatsapp_image(user_number, image_url, caption=caption)
#         except Exception as e:
#             print("Error sending menu image in send_category_buttons:", e)

#     if lang == "ar":
#         body = "من فضلك اختر الفئة:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "برجر ووجبات"},
#             {"id": "cat_sandwiches", "title": "ساندويتش"},
#             {"id": "cat_snacks_sides", "title": "وجبات خفيفة"},
#         ]
#     else:
#         body = "Please choose a category:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "Burgers & Meals"},
#             {"id": "cat_sandwiches", "title": "Sandwiches & Wraps"},
#             {"id": "cat_snacks_sides", "title": "Snacks & Sides"},
#         ]

#     send_whatsapp_quick_buttons(user_number, body, buttons)


# def send_items_for_category(user_number: str, category: str, lang: str = "en"):
#     items = get_items_by_category(category)
#     if not items:
#         send_whatsapp_text(user_number, "No items found in this category yet.")
#         return

#     state = WA_CATEGORY_STATE.get(user_number, {"category": category, "index": 0})
#     index = state.get("index", 0)

#     if state.get("category") != category:
#         index = 0

#     slice_items = items[index : index + 2]
#     buttons = []

#     for item in slice_items:
#         en_name = item["en"]
#         ar_name = item["ar"]

#         if lang == "ar":
#             title = ar_name
#         else:
#             title = en_name.title()

#         buttons.append({"id": f"item_{en_name}", "title": title})

#     if index + 2 < len(items):
#         if lang == "ar":
#             more_title = "المزيد من الأصناف"
#         else:
#             more_title = "More items"
#         buttons.append({"id": "more_items", "title": more_title})

#     if lang == "ar":
#         body = "من فضلك اختر الصنف (سيتم إضافة 1 كمية لكل ضغطة):"
#     else:
#         body = "Please choose an item (1 quantity will be added per click):"

#     send_whatsapp_quick_buttons(user_number, body, buttons)
#     WA_CATEGORY_STATE[user_number] = {"category": category, "index": index}


# # ---------------------------
# # LLM Reply
# # ---------------------------


# def get_llm_reply(msg, lang="en"):
#     lang_name = "English" if lang == "en" else "Arabic"

#     sys_prompt = (
#         "You are Joana Fast Food Assistant.\n"
#         "- Short, clear, polite answers.\n"
#         "- Can answer non-restaurant questions briefly, then bring user back to Joana.\n"
#         "- Joana Fast Food is 24/7, takeaway only.\n"
#         f"Always respond in {lang_name}.\n"
#     )

#     context = build_menu_context()

#     messages = [
#         {"role": "system", "content": sys_prompt},
#         {"role": "system", "content": context},
#     ]

#     for m in session.get("messages", []):
#         messages.append(m)

#     messages.append({"role": "user", "content": msg})

#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=messages,
#             temperature=0.5,
#             max_tokens=250,
#         )
#         return res.choices[0].message.content.strip()

#     except Exception as e:
#         print("LLM error:", e)
#         return "Sorry, something went wrong." if lang == "en" else "عذراً، حدث خطأ ما."


# # ---------------------------
# # Routes
# # ---------------------------


# @app.route("/")
# def index():
#     return render_template("index.html")


# @app.route("/cloud-test")
# def cloud_test():
#     # yahan apna khud ka number daal kar test kar sakti ho
#     send_whatsapp_text("9230XXXXXXXX", "Hello from Joana bot via WhatsApp Cloud API!")
#     return "Cloud test message sent!"


# # ---------------------------
# # WhatsApp Cloud Webhook
# # ---------------------------

# @app.route("/whatsapp/webhook", methods=["GET", "POST"])
# def whatsapp_webhook():
#     # 1) Verification handshake (GET)
#     if request.method == "GET":
#         mode = request.args.get("hub.mode")
#         token = request.args.get("hub.verify_token")
#         challenge = request.args.get("hub.challenge")

#         print("Webhook verification:", mode, token, challenge)

#         if mode == "subscribe" and token == WHATSAPP_VERIFY_TOKEN:
#             return challenge or "", 200
#         else:
#             return "Verification token mismatch", 403

#     # 2) Incoming messages (POST)
#     data = request.get_json(force=True) or {}
#     try:
#         print("Incoming from Cloud API:", json.dumps(data, indent=2, ensure_ascii=False))
#     except Exception:
#         print("Incoming from Cloud API (raw):", data)

#     user_text = None
#     user_number = None
#     btn_id = None  # 🔹 yahan id store karenge

#     try:
#         entry = (data.get("entry") or [])[0]
#         changes = (entry.get("changes") or [])[0]
#         value = changes.get("value") or {}
#     except Exception:
#         value = {}

#     # Status callbacks (delivery/read) ignore
#     if value.get("statuses"):
#         print("Cloud API status callback – ignoring")
#         return "ok", 200

#     messages = value.get("messages")
#     if not messages:
#         print("No messages in payload.")
#         return "ignored", 200

#     msg_obj = messages[0]
#     user_number = normalize_number(msg_obj.get("from"))
#     msg_type = msg_obj.get("type")
#     print("Cloud msg_type:", msg_type)

#     if msg_type == "text":
#         user_text = msg_obj.get("text", {}).get("body", "")
#     elif msg_type == "interactive":
#         interactive = msg_obj.get("interactive", {}) or {}
#         itype = interactive.get("type")
#         print("Interactive payload:", interactive)

#         if itype == "button_reply":
#             br = interactive.get("button_reply", {}) or {}
#             btn_id = (br.get("id") or "").strip()
#             user_text = (br.get("title") or btn_id or "").strip()
#             print("Button reply => id:", btn_id, "| title:", user_text)
#         elif itype == "list_reply":
#             lr = interactive.get("list_reply", {}) or {}
#             btn_id = (lr.get("id") or "").strip()
#             user_text = (lr.get("title") or btn_id or "").strip()
#             print("List reply => id:", btn_id, "| title:", user_text)
#     else:
#         # other types (image, audio, etc.) -> simple placeholder
#         user_text = ""

#     if not user_number or not user_text:
#         print("Could not parse user_number or user_text from Cloud payload.")
#         return "ignored", 200

#     print("Parsed RAW:", user_number, "| text:", repr(user_text), "| id:", repr(btn_id))
#     if "\n" in user_text:
#         parts = [p.strip() for p in user_text.split("\n") if p.strip()]
#         if parts:
#             user_text = parts[-1]
#     print("Parsed CLEANED:", user_number, "| text:", repr(user_text))

#     clean = user_text.strip().lower()
#     clean_id = (btn_id or "").strip().lower()   # 🔹 id ka lowercase

#     lang = detect_language(user_text or "")
#     if any("\u0600" <= ch <= "\u06ff" for ch in user_text):
#         lang = "ar"

#     last_state = WHATSAPP_SESSIONS.get(user_number, {}).get("state", {}) or {}
#     last_stage = last_state.get("stage")

#     # 1️⃣ Greeting -> branches + "Order via text / voice"
#     if last_stage is None and is_wa_greeting(user_text):
#         if lang == "ar":
#             buttons = [
#                 {"id": "order_text", "title": "الطلب عبر الرسائل"},
#                 {"id": "order_voice", "title": "الطلب عبر الصوت"},
#             ]
#             body = build_branches_message("ar")
#         else:
#             buttons = [
#                 {"id": "order_text", "title": "Order via text"},
#                 {"id": "order_voice", "title": "Order via voice"},
#             ]
#             body = build_branches_message("en")

#         send_whatsapp_quick_buttons(user_number, body, buttons)
#         return "ok", 200

#     # 2️⃣ Category selection (title + id dono se match)
#     if (
#         clean in ["burgers & meals", "burgers_meals", "برجر ووجبات"]
#         or clean_id == "cat_burgers_meals"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "burgers_meals", "index": 0}
#         send_items_for_category(user_number, "burgers_meals", lang)
#         return "ok", 200

#     if (
#         clean in ["sandwiches & wraps", "sandwiches", "ساندويتش", "ساندويتشات"]
#         or clean_id == "cat_sandwiches"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "sandwiches", "index": 0}
#         send_items_for_category(user_number, "sandwiches", lang)
#         return "ok", 200

#     if (
#         clean in ["snacks & sides", "snacks_sides", "وجبات خفيفة"]
#         or clean_id == "cat_snacks_sides"
#     ):
#         WA_CATEGORY_STATE[user_number] = {"category": "snacks_sides", "index": 0}
#         send_items_for_category(user_number, "snacks_sides", lang)
#         return "ok", 200

#     # 3️⃣ More items in category  👉 yahan main fix
#     if (
#         clean in ["more items", "المزيد من الأصناف"]    # exact title
#         or ("more" in clean and "item" in clean)        # agar space / case thora diff ho
#         or clean_id == "more_items"                     # button id
#     ) and user_number in WA_CATEGORY_STATE:
#         st = WA_CATEGORY_STATE.get(user_number, {})
#         cat = st.get("category")
#         if cat:
#             items = get_items_by_category(cat)
#             if items:
#                 new_index = st.get("index", 0) + 2
#                 if new_index >= len(items):
#                     new_index = 0
#                 WA_CATEGORY_STATE[user_number] = {"category": cat, "index": new_index}
#                 send_items_for_category(user_number, cat, lang)
#                 return "ok", 200

#     # 4️⃣ Add more / finish order buttons (from interactive)
#     if (
#         clean in ["add more items", "إضافة أصناف أخرى", "اضافة اصناف اخرى"]
#         or clean_id in ["add_more_items", "payment_add_more"]
#     ) and last_stage in ["add_more", "payment"]:
#         sess = WHATSAPP_SESSIONS.get(user_number)
#         if isinstance(sess, dict):
#             st = sess.get("state") or {}
#             st["stage"] = "add_more"
#             sess["state"] = st
#             WHATSAPP_SESSIONS[user_number] = sess

#         send_category_buttons(user_number, lang, show_image=False)
#         return "ok", 200

#     if clean in ["finish order", "إنهاء الطلب", "انهاء الطلب"] or clean_id == "finish_order":
#         user_text = "done"
#         clean = "done"

#     # 5️⃣ "Order via text" button
#     if clean_id == "order_text" or clean in [
#         "1",
#         "1.",
#         "order via text",
#         "order by text",
#         "text order",
#         "الطلب عبر الرسائل",
#         "الطلب بالرسائل",
#         "اطلب بالرسائل",
#     ]:
#         send_category_buttons(user_number, lang, show_image=True)
#         return "ok", 200

#     # 6️⃣ "Order via voice"
#     if clean_id == "order_voice" or clean in [
#         "2",
#         "2.",
#         "order via voice",
#         "order by voice",
#         "voice order",
#         "voice",
#         "الطلب عبر الصوت",
#         "الطلب بالصوت",
#         "اطلب بالصوت",
#     ]:
#         send_whatsapp_text(
#             user_number,
#             "🎙 من فضلك أرسل طلبك على شكل رسالة صوتية.\n\n"
#             'مثال: "اثنين زنجر برجر، واحد حار وواحد بدون حار، وبطاطس كبيرة".\n\n'
#             "سيقوم الموظف بالاستماع وتأكيد طلبك 😊",
#         )
#         return "ok", 200

#     # 7️⃣ Baaki sab normal chatbot flow (aap ka purana code yahan se same)
#     try:
#         result = process_whatsapp_message(user_number, user_text, lang)

#         reply_html = result.get("reply") or "Sorry, something went wrong."
#         reply_text = html_to_whatsapp(reply_html)

#         stage = result.get("stage")
#         order = result.get("order") or []

#         if result.get("menu"):
#             image_url = (
#                 "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#             )
#             send_whatsapp_image(user_number, image_url, caption="Here is our menu 📋")

#         # Spicy selection buttons
#         if stage == "await_spice":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "spice_spicy", "title": "حار"},
#                     {"id": "spice_non_spicy", "title": "بدون حار"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "spice_spicy", "title": "Spicy"},
#                     {"id": "spice_non_spicy", "title": "Non-spicy"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Add more / finish order buttons (after adding an item)
#         if stage == "add_more" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
#                     {"id": "finish_order", "title": "إنهاء الطلب"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "add_more_items", "title": "Add more items"},
#                     {"id": "finish_order", "title": "Finish order"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Payment page buttons
#         if stage == "payment" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "payment_proceed", "title": "المتابعة إلى الدفع"},
#                     {"id": "payment_add_more", "title": "إضافة أصناف أخرى"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "payment_proceed", "title": "Proceed to payment"},
#                     {"id": "payment_add_more", "title": "Add more items"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         # Choose payment method buttons
#         if stage == "choose_payment":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "pay_online", "title": "الدفع عبر الإنترنت"},
#                     {"id": "pay_cash", "title": "الدفع نقداً"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "pay_online", "title": "Online payment"},
#                     {"id": "pay_cash", "title": "Cash"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#     except Exception as e:
#         print("Error in whatsapp_webhook:", e)
#         reply_text = "Sorry, something went wrong."

#     send_whatsapp_text(user_number, reply_text)
#     return "ok", 200



# def process_whatsapp_message(
#     user_number: str, user_text: str, lang_hint: str = None
# ) -> dict:
#     ctx = WHATSAPP_SESSIONS.get(user_number, {})
#     prev_state = ctx.get("state")
#     prev_messages = ctx.get("messages")

#     with app.test_request_context(
#         "/api/chat",
#         method="POST",
#         json={
#             "message": user_text,
#             "is_voice": False,
#             "lang_hint": lang_hint,
#         },
#     ):
#         if prev_state is not None:
#             session["state"] = prev_state
#         if prev_messages is not None:
#             session["messages"] = prev_messages

#         resp = chat()

#         WHATSAPP_SESSIONS[user_number] = {
#             "state": session.get("state"),
#             "messages": session.get("messages"),
#         }

#         if isinstance(resp, tuple):
#             resp_obj = resp[0]
#         else:
#             resp_obj = resp

#         data = resp_obj.get_json() or {}
#         return data


# # ---------------------------
# # Web chat route
# # ---------------------------


# def make_chat_response(reply, lang, menu=None):
#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     payload = {
#         "reply": reply,
#         "lang": lang,
#         "stage": s.get("stage"),
#         "order": s.get("order"),
#         "total": s.get("total", 0),
#     }
#     if menu:
#         payload["menu"] = menu
#     return jsonify(payload)


# @app.route("/api/chat", methods=["POST"])
# def chat():
#     global MENU, BRANCHES
#     MENU = load_menu()
#     BRANCHES = load_branches()

#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     session["state"] = s
#     session["messages"] = session.get("messages", [])

#     data = request.get_json(force=True)
#     msg_raw = (data.get("message") or "").strip()
#     msg = msg_raw
#     is_voice = data.get("is_voice", False)
#     lang_hint = data.get("lang_hint")

#     lang = detect_language(msg_raw)

#     if any("\u0600" <= ch <= "\u06ff" for ch in msg_raw):
#         lang = "ar"

#     if lang_hint in ("en", "ar"):
#         lang = lang_hint

#     if lang == "ar":
#         msg = llm_normalize_text(msg_raw, lang)

#     intent = detect_intent(msg)

#     session["messages"].append({"role": "user", "content": msg})

#     msg_l = msg.lower()
#     if "menu" in msg_l or "send menu" in msg_l or "price list" in msg_l:
#         intent = "menu"

#     if polite_check(msg):
#         reply = "Please speak politely." if lang == "en" else "من فضلك تحدث بأدب."
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     AR_GREETINGS = [
#         "مرحبا",
#         "مرحبا بك",
#         "اهلا",
#         "أهلا",
#         "اهلاً",
#         "اهلا وسهلا",
#         "السلام عليكم",
#         "السلام",
#         "هلا",
#         "يا هلا",
#         "مساء الخير",
#         "صباح الخير",
#     ]

#     EN_GREETINGS = [
#         "hi",
#         "hello",
#         "hey",
#         "good morning",
#         "good evening",
#         "good afternoon",
#         "hiya",
#         "yo",
#     ]

#     def is_greeting_text(msg_: str, lang_: str) -> bool:
#         """
#         Sirf tab greeting ho jab pure text me sirf greeting hi ho.
#         'chicken sandwich' jese text ko greeting na samjho.
#         """
#         if not msg_:
#             return False

#         text = msg_.strip().lower()
#         text = re.sub(r"[^\w\u0600-\u06FF ]+", "", text)

#         greetings = AR_GREETINGS if lang_ == "ar" else EN_GREETINGS
#         return text in greetings

#     # Greeting (only if stage empty)
#     if s.get("stage") is None and is_greeting_text(msg, lang):
#         s = {
#             "stage": "branch",
#             "order": [],
#             "total": 0,
#             "last_item": None,
#             "last_qty": 0,
#         }
#         session["state"] = s

#         if lang == "ar":
#             reply = (
#                 "مرحباً بك في مطعم جوانا للوجبات السريعة! 🍔<br>"
#                 "ابحث عن أقرب فرع واتصل على الرقم للطلب:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>جاهز للطلب؟ شارك طلبك بالصوت أو بالكتابة!<br><br>"
#             )
#         else:
#             reply = (
#                 "Welcome to JOANA Fast Food! <br>"
#                 "Find your nearest branch and dial the number to order:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>Ready to order? Share your order via voice or chat!<br><br>"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     if intent == "menu":
#         reply = (
#             "Here’s our menu! Please place your order."
#             if lang == "en"
#             else "هذه قائمتنا! من فضلك ضع طلبك."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang, menu="/static/menu.PNG")

#     # user says "add more" in payment (free text)
#     if s.get("stage") == "payment" and (
#         "add more" in msg.lower() or "إضافة" in msg or "اضافة" in msg or "أخرى" in msg
#     ):
#         s["stage"] = "add_more"
#         session["state"] = s
#         reply = (
#             "Sure! Please tell me what else you would like to add to your order."
#             if lang == "en"
#             else "حسناً! من فضلك أخبرني ما الذي تريد إضافته أيضاً."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     # Spicy follow-up
#     if s.get("stage") == "await_spice" and s.get("last_item"):
#         last_item = s["last_item"]
#         last_qty = s.get("last_qty", 1)
#         price, _category = get_price_and_category(last_item)
#         split = parse_spice_split(msg)

#         if split:
#             spicy_q, non_q = split
#         else:
#             spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#             if spicy_flag:
#                 spicy_q, non_q = last_qty, 0
#             elif nonspicy_flag:
#                 spicy_q, non_q = 0, last_qty
#             else:
#                 spicy_q, non_q = 0, last_qty

#         order_added_msgs = []

#         if spicy_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": spicy_q,
#                     "spicy": 1,
#                     "nonspicy": 0,
#                     "price": price,
#                     "subtotal": spicy_q * price,
#                 }
#             )
#             order_added_msgs.append(f"{spicy_q} {'spicy' if lang == 'en' else 'حار'}")

#         if non_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": non_q,
#                     "spicy": 0,
#                     "nonspicy": 1,
#                     "price": price,
#                     "subtotal": non_q * price,
#                 }
#             )
#             order_added_msgs.append(
#                 f"{non_q} {'non-spicy' if lang == 'en' else 'بدون حار'}"
#             )

#         s["stage"] = "add_more"
#         s["last_item"] = None
#         s["last_qty"] = 0
#         session["state"] = s

#         reply = (
#             f"Added {last_item.title()} — "
#             + " & ".join(order_added_msgs)
#             + ". Would you like to add anything else?"
#             if lang == "en"
#             else f"تمت إضافة {last_item} — "
#             + " و ".join(order_added_msgs)
#             + ". هل ترغب في إضافة شيء آخر؟"
#         )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Item detection
#     found = find_menu_item(msg)

#     if found:
#         qty = detect_qty(msg)
#         price, category = get_price_and_category(found)

#         s["last_item"] = found
#         s["last_qty"] = qty
#         session["state"] = s

#         spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#         if category == "burgers_meals" and not (spicy_flag or nonspicy_flag):
#             s["stage"] = "await_spice"
#             session["state"] = s

#             reply = (
#                 f"Would you like your {found.title()} spicy or non-spicy?"
#                 if lang == "en"
#                 else f"هل ترغب أن يكون {found} حارًا أم بدون حار؟"
#             )

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         if category == "burgers_meals":
#             if spicy_flag:
#                 spicy = 1
#                 nonspicy = 0
#             elif nonspicy_flag:
#                 spicy = 0
#                 nonspicy = 1
#             else:
#                 spicy = 0
#                 nonspicy = 1
#         else:
#             spicy = 0
#             nonspicy = 0

#         s["order"].append(
#             {
#                 "item": found,
#                 "qty": qty,
#                 "spicy": spicy,
#                 "nonspicy": nonspicy,
#                 "price": price,
#                 "subtotal": qty * price,
#             }
#         )

#         previous_stage = s.get("stage")
#         s["last_item"] = None
#         s["last_qty"] = 0

#         if previous_stage == "payment":
#             summary, total = build_order_summary_and_total(s["order"], lang)
#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             reply = (
#                 "Your order has been updated! Here's the new summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تحديث طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )
#         else:
#             s["stage"] = "add_more"
#             session["state"] = s

#             reply = (
#                 f"{found.title()} ×{qty} added! Would you like to add anything else?"
#                 if lang == "en"
#                 else f"تمت إضافة {found} ×{qty}! هل ترغب في إضافة شيء آخر؟"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Done / no more items
#     if any(x in msg.lower() for x in ["no", "done", "that's all", "that all"]) or any(
#         x in msg for x in ["خلص", "لا", "انتهى", "كفاية"]
#     ):
#         if s.get("order"):

#             summary, total = build_order_summary_and_total(s["order"], lang)

#             reply = (
#                 "Your order is confirmed! Here's a summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تأكيد طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Payment yes
#     if s.get("stage") == "payment" and (
#         any(
#             x in msg.lower()
#             for x in [
#                 "yes",
#                 "sure",
#                 "ok",
#                 "okay",
#                 "proceed",
#                 "go ahead",
#             ]
#         )
#         or any(
#             x in msg
#             for x in [
#                 "تمام",
#                 "نعم",
#                 "أكيد",
#                 "المتابعة",
#                 "تابع",
#                 "ادفع",
#                 "إلى الدفع",
#             ]
#         )
#     ):
#         reply = (
#             "How would you like to pay?<br><b>Cash</b> or <b>Online Payment</b>?"
#             if lang == "en"
#             else "كيف ترغب في الدفع؟<br><b> نقدي </b> أو <b> الدفع عبر الإنترنت</b>؟"
#         )
#         s["stage"] = "choose_payment"
#         session["state"] = s

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Choose payment
#     if s.get("stage") == "choose_payment":
#         total = s.get("total", 0)
#         msg_l = msg.lower()

#         if any(x in msg_l for x in ["cash", "cod", "نقد", "نقدا"]):
#             reply = (
#                 "Thank you for your order! Your order will be ready in 20 to 30 minutes."
#                 if lang == "en"
#                 else "شكراً لطلبك! سيكون جاهزاً خلال 20 إلى 30 دقيقة."
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         elif any(
#             x in msg_l
#             for x in [
#                 "online",
#                 "online payment",
#                 "pay online",
#                 "card",
#                 "visa",
#                 "master",
#                 "debit",
#                 "mada",
#             ]
#         ) or any(x in msg for x in ["مدى", "إلكتروني", "دفع", "كرت"]):
#             reply = (
#                 f"Great! Your payment of {total:.2f} {CURRENCY} is being processed.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>Click here to complete your payment</b></a>"
#                 if lang == "en"
#                 else f"رائع! يتم الآن معالجة دفعتك بمقدار {total:.2f} {CURRENCY_AR}.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>اضغط هنا لإتمام الدفع</b></a>"
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         else:
#             reply = (
#                 "Please select a valid payment method — Cash or Online."
#                 if lang == "en"
#                 else "يرجى اختيار طريقة الدفع الصحيحة - نقدًا أو عبر الإنترنت"
#             )
#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Generic LLM reply
#     reply = get_llm_reply(msg, lang)

#     if s.get("stage") == "payment" and s.get("order"):
#         summary, total = build_order_summary_and_total(s["order"], lang)
#         s["total"] = total
#         session["state"] = s

#         if lang == "en":
#             reply = (
#                 reply
#                 + "<br><br>Here is your current order summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#             )
#         else:
#             reply = (
#                 reply
#                 + "<br><br>هذا ملخص طلبك الحالي:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#     if is_voice:
#         speak_text(reply, lang)

#     return make_chat_response(reply, lang)


# # ---------------------------
# # Run App (local)
# # ---------------------------
# if __name__ == "__main__":
#     app.run(debug=True)

# import json
# import os
# import re
# import sqlite3
# import requests

# import pandas as pd
# from flask import Flask, render_template, request, jsonify, session
# from nlp_utils import detect_intent, detect_language
# from openai import OpenAI
# from dotenv import load_dotenv

# load_dotenv()

# # ---------------------------
# # ENV VARIABLES
# # ---------------------------
# OPENROUTER_API_KEY = os.getenv("OPENROUTER_API_KEY", "YOUR_OPENROUTER_API_KEY_HERE")
# FLASK_SECRET = os.getenv("FLASK_SECRET_KEY", "joana-fastfood-secret")

# # WhatsApp Cloud API envs
# WHATSAPP_TOKEN = os.getenv("WHATSAPP_TOKEN")  # permanent token (no "Bearer " in .env)
# WHATSAPP_PHONE_NUMBER_ID = os.getenv("WHATSAPP_PHONE_NUMBER_ID")
# WHATSAPP_VERIFY_TOKEN = os.getenv("WHATSAPP_VERIFY_TOKEN", "joana-verify-token-123")

# WHATSAPP_API_BASE = "https://graph.facebook.com/v21.0"

# # ---------------------------
# # WHATSAPP CLOUD SEND HELPERS
# # ---------------------------


# def send_whatsapp_text(to_number: str, text: str):
#     print("=== send_whatsapp_text (Cloud API) CALLED ===")
#     print("WHATSAPP_TOKEN present?", bool(WHATSAPP_TOKEN))
#     print("Sending to:", to_number, "| Text:", text)

#     if not WHATSAPP_TOKEN or not WHATSAPP_PHONE_NUMBER_ID:
#         print("WhatsApp Cloud API config missing, cannot send WhatsApp message.")
#         return

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "recipient_type": "individual",
#         "to": to_number,
#         "type": "text",
#         "text": {"body": text},
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD API TEXT SEND RESPONSE:", resp.status_code, resp.text)
#     except Exception as e:
#         print("Error sending WhatsApp message via Cloud API:", repr(e))


# def send_whatsapp_image(to_number: str, image_url: str, caption: str = ""):
#     if not WHATSAPP_TOKEN or not WHATSAPP_PHONE_NUMBER_ID:
#         print("WhatsApp Cloud API config missing, cannot send WhatsApp image.")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }
#     payload = {
#         "messaging_product": "whatsapp",
#         "recipient_type": "individual",
#         "to": to_number,
#         "type": "image",
#         "image": {
#             "link": image_url,
#         },
#     }
#     if caption:
#         payload["image"]["caption"] = caption

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=30)
#         print("CLOUD API IMAGE SEND:", resp.status_code, resp.text)
#         return 200 <= resp.status_code < 300
#     except Exception as e:
#         print("Error sending WhatsApp image via Cloud API:", e)
#         return False


# def send_whatsapp_quick_buttons(to_number: str, body_text: str, buttons: list):
#     if not WHATSAPP_TOKEN or not WHATSAPP_PHONE_NUMBER_ID:
#         print("WhatsApp Cloud API config missing, cannot send WhatsApp buttons.")
#         return False

#     url = f"{WHATSAPP_API_BASE}/{WHATSAPP_PHONE_NUMBER_ID}/messages"
#     headers = {
#         "Authorization": f"Bearer {WHATSAPP_TOKEN}",
#         "Content-Type": "application/json",
#     }

#     cloud_buttons = []
#     for b in buttons[:3]:
#         cloud_buttons.append(
#             {
#                 "type": "reply",
#                 "reply": {
#                     "id": b.get("id", b["title"]),
#                     "title": b["title"],
#                 },
#             }
#         )

#     payload = {
#         "messaging_product": "whatsapp",
#         "to": to_number,
#         "type": "interactive",
#         "interactive": {
#             "type": "button",
#             "body": {"text": body_text},
#             "action": {"buttons": cloud_buttons},
#         },
#     }

#     try:
#         resp = requests.post(url, headers=headers, json=payload, timeout=10)
#         print("CLOUD API BUTTONS SEND:", resp.status_code, resp.text)
#         return 200 <= resp.status_code < 300
#     except Exception as e:
#         print("Error sending WhatsApp buttons via Cloud API:", e)
#         return False


# # ---------------------------
# # Paths / Directories
# # ---------------------------
# BASE_DIR = os.path.dirname(os.path.abspath(__file__))
# DATA_DIR = os.path.join(BASE_DIR, "data")

# MENU_FILE = os.path.join(DATA_DIR, "Menu.xlsx")
# BRANCHES_FILE = os.path.join(DATA_DIR, "Branches.xlsx")

# WHATSAPP_SESSIONS = {}
# WA_CATEGORY_STATE = {}  # phone -> {"category": str, "index": int}

# # ---------------------------
# # Flask Config
# # ---------------------------
# app = Flask(__name__, static_folder="static", template_folder="templates")
# app.secret_key = FLASK_SECRET

# client = OpenAI(api_key=OPENROUTER_API_KEY, base_url="https://openrouter.ai/api/v1")

# PAYMENT_URL = (
#     "https://starlit-sopapillas-520aa2.netlify.app/"
#     "?redirect=http://127.0.0.1:5000/thankyou"
# )
# CURRENCY = "SAR"
# CURRENCY_AR = "ريال سعودي"

# # ---------------------------
# # Database
# # ---------------------------
# DB_PATH = os.path.join(DATA_DIR, "orders.db")


# def init_db():
#     os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
#     conn = sqlite3.connect(DB_PATH)
#     conn.execute(
#         """
#         CREATE TABLE IF NOT EXISTS orders (
#             id INTEGER PRIMARY KEY AUTOINCREMENT,
#             item TEXT,
#             qty INTEGER,
#             spicy INTEGER,
#             nonspicy INTEGER,
#             price REAL,
#             subtotal REAL,
#             total REAL,
#             payment_method TEXT,
#             datetime TEXT
#         )
#         """
#     )
#     conn.commit()
#     conn.close()


# init_db()

# # ---------------------------
# # Category mapping helper
# # ---------------------------


# def canonical_category(raw_category: str, name_en: str = "") -> str:
#     """
#     Excel category ko 3 main groups me convert:
#     burgers_meals, sandwiches, snacks_sides
#     """
#     c = (raw_category or "").strip().lower()
#     n = (name_en or "").strip().lower()

#     if c in ["burgers", "burger", "meals", "meal"]:
#         return "burgers_meals"
#     if c == "sandwiches":
#         return "sandwiches"
#     if c in ["sides", "side", "drinks", "drink"]:
#         return "snacks_sides"

#     if any(
#         k in n for k in ["burger", "zinger", "tortilla", "hot dog", "hotdog", "meal"]
#     ):
#         return "burgers_meals"
#     if any(k in n for k in ["sandwich", "kudu", "kabab"]):
#         return "sandwiches"
#     if any(
#         k in n
#         for k in [
#             "fries",
#             "nuggets",
#             "rings",
#             "juice",
#             "corn",
#             "potato",
#             "samosa",
#             "popcorn",
#         ]
#     ):
#         return "snacks_sides"

#     return ""


# # ---------------------------
# # Excel Loaders
# # ---------------------------


# def load_menu():
#     try:
#         df_raw = pd.read_excel(MENU_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             if any("name_en" in str(c).lower() for c in row):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             print("❌ Could not find header row containing 'name_en'")
#             return {}

#         df = pd.read_excel(MENU_FILE, header=header_row_index)

#         def has(col, *keys):
#             c = str(col).strip().lower()
#             return any(k in c for k in keys)

#         english_col = next((c for c in df.columns if has(c, "name_en")), None)
#         arabic_col = next((c for c in df.columns if has(c, "name_ar")), None)
#         price_col = next((c for c in df.columns if has(c, "price")), None)
#         cat_col = next(
#             (c for c in df.columns if has(c, "category", "cat", "type")), None
#         )

#         if not english_col or not price_col:
#             raise Exception("Missing name_en or price column")

#         menu = {}
#         for _, row in df.iterrows():
#             en = str(row[english_col]).strip().lower()
#             if not en or en == "nan":
#                 continue

#             ar = str(row[arabic_col]).strip() if arabic_col else en
#             raw_cat = str(row.get(cat_col, "")).strip() if cat_col else ""

#             try:
#                 price = float(row[price_col])
#             except Exception:
#                 continue

#             cat_final = canonical_category(raw_cat, en)

#             entry = {
#                 "price": price,
#                 "category": cat_final,
#                 "name_en": en,
#                 "name_ar": ar or en,
#             }

#             menu[en] = entry
#             if ar:
#                 menu[ar.lower()] = entry

#         print(f"✅ Loaded {len(menu)} menu items.")
#         return menu

#     except Exception as e:
#         print("Menu load failed:", e)
#         return {}


# def load_branches():
#     try:
#         df_raw = pd.read_excel(BRANCHES_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             row_l = [str(c).lower() for c in row]
#             if any("branch" in c for c in row_l) and any("address" in c for c in row_l):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             header_row_index = 0

#         df = pd.read_excel(BRANCHES_FILE, header=header_row_index)

#         name_col = next((c for c in df.columns if "branch" in str(c).lower()), None)
#         addr_col = next((c for c in df.columns if "address" in str(c).lower()), None)
#         phone_col = next(
#             (
#                 c
#                 for c in df.columns
#                 if "phone" in str(c).lower() or "number" in str(c).lower()
#             ),
#             None,
#         )

#         branches = []
#         for _, row in df.iterrows():
#             branches.append(
#                 {
#                     "Branch Name": str(row.get(name_col, "")).strip(),
#                     "Address / Area": str(row.get(addr_col, "")).strip(),
#                     "Phone Number": str(row.get(phone_col, "")).strip(),
#                 }
#             )

#         print(f"✅ Loaded {len(branches)} branches.")
#         return [
#             b
#             for b in branches
#             if (b["Branch Name"] or b["Address / Area"] or b["Phone Number"])
#         ]

#     except Exception as e:
#         print("❌ Branch load failed:", e)
#         return []


# MENU = load_menu()
# BRANCHES = load_branches()

# # ---------------------------
# # Helpers
# # ---------------------------


# def build_menu_context():
#     if not MENU:
#         return "Current restaurant menu is empty."

#     lines = []
#     for name, info in MENU.items():
#         if not re.search(r"[A-Za-z]", name):
#             continue
#         price = info.get("price", 0.0)
#         category = info.get("category", "") or ""
#         lines.append(f"- {name} | price: {price:.2f} {CURRENCY} | category: {category}")

#     if not lines:
#         return "Current restaurant menu is empty."

#     return "Current restaurant menu (items, prices, categories):\n" + "\n".join(lines)


# def build_branches_message(lang: str = "en") -> str:
#     lines = []
#     for b in BRANCHES[:6]:
#         name = b.get("Branch Name", "")
#         addr = b.get("Address / Area", "")
#         phone = b.get("Phone Number", "")
#         if phone:
#             lines.append(f"- {name} – {addr} | {phone}")

#     branches_text = "\n".join(lines) if lines else "Branches info not available."

#     if lang == "ar":
#         return (
#             "مرحباً بك في مطعم جوانا للوجبات السريعة!\n\n"
#             "الفروع:\n"
#             f"{branches_text}\n\n"
#             "من فضلك اختر طريقة الطلب:"
#         )
#     else:
#         return (
#             "Welcome to JOANA Fast Food! \n\n"
#             "Branches:\n"
#             f"{branches_text}\n\n"
#             "Please choose an option:"
#         )


# def html_to_whatsapp(text: str) -> str:
#     if not text:
#         return ""

#     text = text.replace("<br>", "\n").replace("<br/>", "\n").replace("<br />", "\n")

#     def replace_anchor(match):
#         full_tag = match.group(0)
#         label = match.group(1).strip()

#         href_match = re.search(
#             r'href=[\'"]([^\'"]+)[\'"]', full_tag, flags=re.IGNORECASE
#         )
#         if not href_match:
#             return label

#         url = href_match.group(1).strip()

#         if url.startswith("tel:"):
#             return label

#         return f"{label}: {url}"

#     text = re.sub(
#         r"<a[^>]+>(.*?)</a>", replace_anchor, text, flags=re.IGNORECASE | re.DOTALL
#     )
#     text = re.sub(r"<[^>]+>", "", text)
#     return text.strip()


# def normalize_whapi_number(raw: str) -> str:
#     if not raw:
#         return ""
#     raw = str(raw)
#     if "@" in raw:
#         raw = raw.split("@", 1)[0]
#     if raw.startswith("+"):
#         raw = raw[1:]
#     return raw


# WA_GREETINGS = [
#     "hi",
#     "hello",
#     "hey",
#     "salam",
#     "slam",
#     "asalam",
#     "assalam",
#     "assalam o alaikum",
#     "assalamu alaikum",
#     "السلام عليكم",
#     "مرحبا",
# ]


# def is_wa_greeting(text: str) -> bool:
#     """
#     WhatsApp greeting sirf tab jab pura text hi greeting ho.
#     'chicken sandwich' mein 'hi' aata hai lekin yeh greeting nahi hai.
#     """
#     if not text:
#         return False
#     t = text.strip().lower()
#     t = re.sub(r"[^\w\u0600-\u06FF ]+", "", t)
#     return t in WA_GREETINGS


# def llm_normalize_text(text: str, lang: str) -> str:
#     if not text or lang != "ar":
#         return text
#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=[
#                 {
#                     "role": "system",
#                     "content": (
#                         "You are a spell-correction engine for user messages.\n"
#                         "- Fix Arabic spelling, don't translate.\n"
#                         "- Return only corrected text."
#                     ),
#                 },
#                 {"role": "user", "content": text},
#             ],
#             temperature=0,
#             max_tokens=100,
#         )
#         return res.choices[0].message.content.strip()
#     except Exception as e:
#         print("LLM normalize error:", e)
#         return text


# def detect_qty(msg: str) -> int:
#     if not msg:
#         return 1
#     text = msg.lower().strip()

#     arabic_digit_map = {
#         "٠": "0",
#         "١": "1",
#         "٢": "2",
#         "٣": "3",
#         "٤": "4",
#         "٥": "5",
#         "٦": "6",
#         "٧": "7",
#         "٨": "8",
#         "٩": "9",
#     }
#     for ar, en in arabic_digit_map.items():
#         text = text.replace(ar, en)

#     digit_match = re.search(r"\b(\d+)\b", text)
#     if digit_match:
#         try:
#             q = int(digit_match.group(1))
#             if q > 0:
#                 return q
#         except ValueError:
#             pass

#     number_words_en = {
#         "one": 1,
#         "two": 2,
#         "three": 3,
#         "four": 4,
#         "five": 5,
#         "six": 6,
#         "seven": 7,
#         "eight": 8,
#         "nine": 9,
#         "ten": 10,
#     }
#     for word, value in number_words_en.items():
#         if re.search(r"\b" + re.escape(word) + r"\b", text):
#             return value

#     number_words_ar = {
#         "واحد": 1,
#         "واحدة": 1,
#         "اثنين": 2,
#         "اتنين": 2,
#         "ثنين": 2,
#         "ثلاثة": 3,
#         "ثلاث": 3,
#         "اربعة": 4,
#         "أربعة": 4,
#         "خمسة": 5,
#         "ستة": 6,
#         "سبعة": 7,
#         "ثمانية": 8,
#         "تسعة": 9,
#         "عشرة": 10,
#     }
#     for word, value in number_words_ar.items():
#         if word in text:
#             return value

#     return 1


# # NEW: multi-item split helper (EN + AR separators)
# def split_order_segments(msg: str):
#     """
#     User message ko chhote parts (segments) me todta hai
#     taake har part se ek item detect kiya ja sake.

#     Example:
#     "2 zinger burger, 1 fries and 3 nuggets"
#     -> ["2 zinger burger", "1 fries", "3 nuggets"]
#     """
#     if not msg:
#         return []

#     text = msg.replace("\n", " ")

#     # Common separators (English + Arabic)
#     separators = [
#         ",",
#         " and ",
#         " & ",
#         "،",
#         " و ",
#     ]

#     for sep in separators:
#         text = text.replace(sep, "|")

#     parts = [p.strip() for p in text.split("|") if p.strip()]
#     return parts


# def polite_check(text):
#     bad = ["idiot", "stupid", "حرام", "لعنة", "غبي"]
#     return any(w in text.lower() for w in bad)


# def get_price_and_category(name):
#     entry = MENU.get(name.lower(), {})
#     return float(entry.get("price", 0)), entry.get("category", "")


# def build_order_summary_and_total(order_items, lang):
#     total = sum(i.get("subtotal", 0) for i in order_items)
#     summary = []

#     for i in order_items:
#         item_name = i["item"]
#         item_info = MENU.get(item_name.lower(), {})
#         category = item_info.get("category")

#         if category == "burgers_meals":
#             if lang == "ar":
#                 kind = "حار" if i.get("spicy") else "بدون حار"
#             else:
#                 kind = "spicy" if i.get("spicy") else "non-spicy"
#             summary.append(f"{i['qty']} {kind} {item_name.title()}")
#         else:
#             summary.append(f"{i['qty']} {item_name.title()}")

#     return summary, total


# def speak_text(reply, lang):
#     return


# def parse_spice_split(msg):
#     text = msg.lower().replace("-", " ")
#     nums = re.findall(r"\d+", text)

#     if len(nums) >= 2:
#         n1, n2 = int(nums[0]), int(nums[1])

#         if ("spicy" in text or "حار" in text) and (
#             "non" in text or "mild" in text or "عادي" in text or "بدون" in text
#         ):
#             spicy_first = (
#                 text.find("spicy") < text.find("non") if "non" in text else True
#             )
#             arab_spicy_first = (
#                 text.find("حار") < text.find("عادي") if "عادي" in text else True
#             )

#             return (n1, n2) if (spicy_first or arab_spicy_first) else (n2, n1)

#     return None


# def find_menu_item(msg):
#     text = msg.lower()
#     candidates = []

#     for name in MENU.keys():
#         pattern = r"\b" + re.escape(name) + r"s?\b"
#         if re.search(pattern, text):
#             candidates.append(name)

#     if not candidates:
#         return None

#     candidates.sort(key=len, reverse=True)
#     return candidates[0]


# def get_items_by_category(category: str):
#     """
#     Return unique items for a category:
#     [{"en": "...", "ar": "..."}]
#     """
#     items_map = {}
#     for key, info in MENU.items():
#         if info.get("category") != category:
#             continue
#         en = (info.get("name_en") or key).strip().lower()
#         ar = (info.get("name_ar") or en).strip()
#         items_map[en] = {"en": en, "ar": ar}
#     return list(items_map.values())


# def detect_spicy_nonspicy(msg: str):
#     """
#     Text se spicy / non-spicy flags nikalta hai.
#     """
#     if not msg:
#         return False, False

#     text = msg.lower()

#     nonspicy_keywords_en = [
#         "non spicy",
#         "non-spicy",
#         "no spicy",
#         "without spicy",
#         "without spice",
#         "not spicy",
#         "mild",
#     ]
#     nonspicy_keywords_ar = [
#         "بدون حار",
#         "بدون حر",
#         "عادي",
#     ]

#     nonspicy_flag = any(k in text for k in nonspicy_keywords_en + nonspicy_keywords_ar)

#     spicy_keywords = [
#         "spicy",
#         "hot",
#         "حار",
#     ]
#     spicy_flag = any(k in text for k in spicy_keywords) and not nonspicy_flag

#     return spicy_flag, nonspicy_flag


# def send_category_buttons(user_number: str, lang: str = "en", show_image: bool = True):
#     """
#     Jab user 'Order via text' ya 'Add more items' bole:

#     - show_image=True  -> menu image + category buttons
#     - show_image=False -> sirf category buttons (no image)
#     """

#     caption = "Here is our menu " if lang == "en" else "هذه قائمتنا"

#     if show_image:
#         image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#         try:
#             send_whatsapp_image(user_number, image_url, caption=caption)
#         except Exception as e:
#             print("Error sending menu image in send_category_buttons:", e)

#     if lang == "ar":
#         body = "من فضلك اختر الفئة:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "برجر ووجبات"},
#             {"id": "cat_sandwiches", "title": "ساندويتش"},
#             {"id": "cat_snacks_sides", "title": "وجبات خفيفة"},
#         ]
#     else:
#         body = "Please choose a category:"
#         buttons = [
#             {"id": "cat_burgers_meals", "title": "Burgers & Meals"},
#             {"id": "cat_sandwiches", "title": "Sandwiches & Wraps"},
#             {"id": "cat_snacks_sides", "title": "Snacks & Sides"},
#         ]

#     send_whatsapp_quick_buttons(user_number, body, buttons)


# def send_items_for_category(user_number: str, category: str, lang: str = "en"):
#     items = get_items_by_category(category)
#     if not items:
#         send_whatsapp_text(user_number, "No items found in this category yet.")
#         return

#     state = WA_CATEGORY_STATE.get(user_number, {"category": category, "index": 0})
#     index = state.get("index", 0)

#     if state.get("category") != category:
#         index = 0

#     slice_items = items[index : index + 2]
#     buttons = []

#     for item in slice_items:
#         en_name = item["en"]
#         ar_name = item["ar"]

#         if lang == "ar":
#             title = ar_name
#         else:
#             title = en_name.title()

#         buttons.append({"id": f"item_{en_name}", "title": title})

#     if index + 2 < len(items):
#         more_title = "المزيد من الأصناف" if lang == "ar" else "More items"
#         buttons.append({"id": "more_items", "title": more_title})

#     if lang == "ar":
#         body = "من فضلك اختر الصنف (سيتم إضافة 1 كمية لكل ضغطة):"
#     else:
#         body = "Please choose an item (1 quantity will be added per click):"

#     send_whatsapp_quick_buttons(user_number, body, buttons)
#     WA_CATEGORY_STATE[user_number] = {"category": category, "index": index}


# # ---------------------------
# # LLM Reply
# # ---------------------------


# def get_llm_reply(msg, lang="en"):
#     lang_name = "English" if lang == "en" else "Arabic"

#     sys_prompt = (
#         "You are Joana Fast Food Assistant.\n"
#         "- Short, clear, polite answers.\n"
#         "- Can answer non-restaurant questions briefly, then bring user back to Joana.\n"
#         "- Joana Fast Food is 24/7, takeaway only.\n"
#         f"Always respond in {lang_name}.\n"
#     )

#     context = build_menu_context()

#     messages = [
#         {"role": "system", "content": sys_prompt},
#         {"role": "system", "content": context},
#     ]

#     for m in session.get("messages", []):
#         messages.append(m)

#     messages.append({"role": "user", "content": msg})

#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=messages,
#             temperature=0.5,
#             max_tokens=250,
#         )
#         return res.choices[0].message.content.strip()

#     except Exception as e:
#         print("LLM error:", e)
#         return "Sorry, something went wrong." if lang == "en" else "عذراً، حدث خطأ ما."


# # ---------------------------
# # Routes
# # ---------------------------


# @app.route("/")
# def index():
#     return render_template("index.html")


# @app.route("/wa-test")
# def wa_test():
#     send_whatsapp_text("923017605889", "Hello from Joana bot via Cloud API ")
#     return "Test message sent!"


# @app.route("/wa-test-buttons")
# def wa_test_buttons():
#     TEST_WHATSAPP_NUMBER = "923017605889"
#     buttons = [
#         {"id": "order_text", "title": "Order via text"},
#         {"id": "order_voice", "title": "Order via voice"},
#     ]
#     send_whatsapp_quick_buttons(
#         TEST_WHATSAPP_NUMBER,
#         build_branches_message("en"),
#         buttons,
#     )
#     return "Buttons test sent!"


# # ---------------------------
# # WhatsApp Cloud API Webhook
# # ---------------------------


# @app.route("/whatsapp/webhook", methods=["GET", "POST"])
# def whatsapp_webhook():
#     # --- 1) Verification (GET) ---
#     if request.method == "GET":
#         mode = request.args.get("hub.mode")
#         token = request.args.get("hub.verify_token")
#         challenge = request.args.get("hub.challenge")
#         print("WEBHOOK VERIFY:", mode, token, challenge)

#         if mode == "subscribe" and token == WHATSAPP_VERIFY_TOKEN:
#             return challenge or "", 200
#         return "Verification failed", 403

#     # --- 2) Incoming messages (POST) ---
#     data = request.get_json(force=True) or {}

#     try:
#         print("Incoming from CLOUD API:", json.dumps(data, indent=2, ensure_ascii=False))
#     except Exception:
#         print("Incoming from CLOUD API (raw):", data)

#     user_text = None
#     user_number = None

#     # Cloud API structure: object -> entry[] -> changes[] -> value -> messages[]
#     entry_list = data.get("entry") or []
#     for entry in entry_list:
#         changes = entry.get("changes") or []
#         for change in changes:
#             value = change.get("value") or {}

#             # ignore status callbacks
#             statuses = value.get("statuses") or []
#             if statuses:
#                 print("Cloud status callback – ignoring")
#                 continue

#             messages = value.get("messages") or []
#             if not messages:
#                 continue

#             msg_obj = messages[0]
#             user_number = msg_obj.get("from")
#             msg_type = msg_obj.get("type")
#             print("Cloud msg_type:", msg_type)

#             if msg_type == "text":
#                 user_text = str(msg_obj.get("text", {}).get("body") or "")

#             elif msg_type == "interactive":
#                 interactive = msg_obj.get("interactive", {}) or {}
#                 itype = interactive.get("type")
#                 if itype == "button_reply":
#                     br = interactive.get("button_reply", {}) or {}
#                     user_text = br.get("title") or br.get("id") or ""
#                 elif itype == "list_reply":
#                     lr = interactive.get("list_reply", {}) or {}
#                     user_text = lr.get("title") or lr.get("id") or ""

#             if user_text:
#                 break
#         if user_text:
#             break

#     if not user_number or not user_text:
#         print("Could not parse incoming Cloud payload.")
#         return "ignored", 200

#     user_number = normalize_whapi_number(user_number)

#     print("Parsed RAW:", user_number, "| text:", repr(user_text))
#     if "\n" in user_text:
#         parts = [p.strip() for p in user_text.split("\n") if p.strip()]
#         if parts:
#             user_text = parts[-1]
#     print("Parsed CLEANED:", user_number, "| text:", repr(user_text))

#     clean = user_text.strip().lower()
#     lang = detect_language(user_text or "")
#     if any("\u0600" <= ch <= "\u06ff" for ch in user_text):
#         lang = "ar"

#     last_state = WHATSAPP_SESSIONS.get(user_number, {}).get("state", {}) or {}
#     last_stage = last_state.get("stage")

#     # 1️⃣ Greeting -> branches + "Order via text / voice"
#     if last_stage is None and is_wa_greeting(user_text):
#         if lang == "ar":
#             buttons = [
#                 {"id": "order_text", "title": "الطلب عبر الرسائل"},
#                 {"id": "order_voice", "title": "الطلب عبر الصوت"},
#             ]
#             body = build_branches_message("ar")
#         else:
#             buttons = [
#                 {"id": "order_text", "title": "Order via text"},
#                 {"id": "order_voice", "title": "Order via voice"},
#             ]
#             body = build_branches_message("en")

#         send_whatsapp_quick_buttons(user_number, body, buttons)
#         return "ok", 200

#     # 2️⃣ Category selection
#     if clean in ["burgers & meals", "burgers_meals", "برجر ووجبات"]:
#         WA_CATEGORY_STATE[user_number] = {"category": "burgers_meals", "index": 0}
#         send_items_for_category(user_number, "burgers_meals", lang)
#         return "ok", 200

#     if clean in ["sandwiches & wraps", "sandwiches", "ساندويتش", "ساندويتشات"]:
#         WA_CATEGORY_STATE[user_number] = {"category": "sandwiches", "index": 0}
#         send_items_for_category(user_number, "sandwiches", lang)
#         return "ok", 200

#     if clean in ["snacks & sides", "snacks_sides", "وجبات خفيفة"]:
#         WA_CATEGORY_STATE[user_number] = {"category": "snacks_sides", "index": 0}
#         send_items_for_category(user_number, "snacks_sides", lang)
#         return "ok", 200

#     # 3️⃣ More items in category
#     if clean in ["more items", "المزيد من الأصناف"] and user_number in WA_CATEGORY_STATE:
#         st = WA_CATEGORY_STATE.get(user_number, {})
#         cat = st.get("category")
#         if cat:
#             items = get_items_by_category(cat)
#             if items:
#                 new_index = st.get("index", 0) + 2
#                 if new_index >= len(items):
#                     new_index = 0
#                 WA_CATEGORY_STATE[user_number] = {"category": cat, "index": new_index}
#                 send_items_for_category(user_number, cat, lang)
#                 return "ok", 200

#     # 4️⃣ Add more / finish order buttons
#     if clean in [
#         "add more items",
#         "إضافة أصناف أخرى",
#         "اضافة اصناف اخرى",
#     ] and last_stage in ["add_more", "payment"]:
#         sess = WHATSAPP_SESSIONS.get(user_number)
#         if isinstance(sess, dict):
#             st = sess.get("state") or {}
#             st["stage"] = "add_more"
#             sess["state"] = st
#             WHATSAPP_SESSIONS[user_number] = sess

#         send_category_buttons(user_number, lang, show_image=False)
#         return "ok", 200

#     if clean in ["finish order", "إنهاء الطلب", "انهاء الطلب"]:
#         user_text = "done"
#         clean = "done"

#     # 5️⃣ "Order via text" button
#     if clean in [
#         "1",
#         "1.",
#         "order via text",
#         "order by text",
#         "text order",
#         "الطلب عبر الرسائل",
#         "الطلب بالرسائل",
#         "اطلب بالرسائل",
#     ]:
#         send_category_buttons(user_number, lang, show_image=True)
#         return "ok", 200

#     # 6️⃣ "Order via voice"
#     if clean in [
#         "2",
#         "2.",
#         "order via voice",
#         "order by voice",
#         "voice order",
#         "voice",
#         "الطلب عبر الصوت",
#         "الطلب بالصوت",
#         "اطلب بالصوت",
#     ]:
#         send_whatsapp_text(
#             user_number,
#             "🎙 من فضلك أرسل طلبك على شكل رسالة صوتية.\n\n"
#             'مثال: "اثنين زنجر برجر، واحد حار وواحد بدون حار، وبطاطس كبيرة".\n\n'
#             "سيقوم الموظف بالاستماع وتأكيد طلبك 😊",
#         )
#         return "ok", 200

#     # 7️⃣ Baaki sab normal chatbot flow
#     try:
#         result = process_whatsapp_message(user_number, user_text, lang)

#         reply_html = result.get("reply") or "Sorry, something went wrong."
#         reply_text = html_to_whatsapp(reply_html)

#         stage = result.get("stage")
#         order = result.get("order") or []

#         if result.get("menu"):
#             image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#             send_whatsapp_image(user_number, image_url, caption="Here is our menu 📋")

#         if stage == "await_spice":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "spice_spicy", "title": "حار"},
#                     {"id": "spice_non_spicy", "title": "بدون حار"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "spice_spicy", "title": "Spicy"},
#                     {"id": "spice_non_spicy", "title": "Non-spicy"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         if stage == "add_more" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "add_more_items", "title": "إضافة أصناف أخرى"},
#                     {"id": "finish_order", "title": "إنهاء الطلب"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "add_more_items", "title": "Add more items"},
#                     {"id": "finish_order", "title": "Finish order"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         if stage == "payment" and order:
#             if lang == "ar":
#                 buttons = [
#                     {"id": "payment_proceed", "title": "المتابعة إلى الدفع"},
#                     {"id": "payment_add_more", "title": "إضافة أصناف أخرى"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "payment_proceed", "title": "Proceed to payment"},
#                     {"id": "payment_add_more", "title": "Add more items"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#         if stage == "choose_payment":
#             if lang == "ar":
#                 buttons = [
#                     {"id": "pay_online", "title": "الدفع عبر الإنترنت"},
#                     {"id": "pay_cash", "title": "الدفع نقداً"},
#                 ]
#             else:
#                 buttons = [
#                     {"id": "pay_online", "title": "Online payment"},
#                     {"id": "pay_cash", "title": "Cash"},
#                 ]
#             send_whatsapp_quick_buttons(user_number, reply_text, buttons)
#             return "ok", 200

#     except Exception as e:
#         print("Error in whatsapp_webhook:", e)
#         reply_text = "Sorry, something went wrong."

#     send_whatsapp_text(user_number, reply_text)
#     return "ok", 200


# def process_whatsapp_message(
#     user_number: str, user_text: str, lang_hint: str = None
# ) -> dict:
#     ctx = WHATSAPP_SESSIONS.get(user_number, {})
#     prev_state = ctx.get("state")
#     prev_messages = ctx.get("messages")

#     with app.test_request_context(
#         "/api/chat",
#         method="POST",
#         json={
#             "message": user_text,
#             "is_voice": False,
#             "lang_hint": lang_hint,
#         },
#     ):
#         if prev_state is not None:
#             session["state"] = prev_state
#         if prev_messages is not None:
#             session["messages"] = prev_messages

#         resp = chat()

#         WHATSAPP_SESSIONS[user_number] = {
#             "state": session.get("state"),
#             "messages": session.get("messages"),
#         }

#         if isinstance(resp, tuple):
#             resp_obj = resp[0]
#         else:
#             resp_obj = resp

#         data = resp_obj.get_json() or {}
#         return data


# # ---------------------------
# # Web chat route
# # ---------------------------


# def make_chat_response(reply, lang, menu=None):
#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     payload = {
#         "reply": reply,
#         "lang": lang,
#         "stage": s.get("stage"),
#         "order": s.get("order"),
#         "total": s.get("total", 0),
#     }
#     if menu:
#         payload["menu"] = menu
#     return jsonify(payload)


# @app.route("/api/chat", methods=["POST"])
# def chat():
#     global MENU, BRANCHES
#     MENU = load_menu()
#     BRANCHES = load_branches()

#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     session["state"] = s
#     session["messages"] = session.get("messages", [])

#     data = request.get_json(force=True)
#     msg_raw = (data.get("message") or "").strip()
#     msg = msg_raw
#     is_voice = data.get("is_voice", False)
#     lang_hint = data.get("lang_hint")

#     lang = detect_language(msg_raw)

#     if any("\u0600" <= ch <= "\u06ff" for ch in msg_raw):
#         lang = "ar"

#     if lang_hint in ("en", "ar"):
#         lang = lang_hint

#     if lang == "ar":
#         msg = llm_normalize_text(msg_raw, lang)

#     intent = detect_intent(msg)

#     session["messages"].append({"role": "user", "content": msg})

#     msg_l = msg.lower()
#     if "menu" in msg_l or "send menu" in msg_l or "price list" in msg_l:
#         intent = "menu"

#     if polite_check(msg):
#         reply = "Please speak politely." if lang == "en" else "من فضلك تحدث بأدب."
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     AR_GREETINGS = [
#         "مرحبا",
#         "مرحبا بك",
#         "اهلا",
#         "أهلا",
#         "اهلاً",
#         "اهلا وسهلا",
#         "السلام عليكم",
#         "السلام",
#         "هلا",
#         "يا هلا",
#         "مساء الخير",
#         "صباح الخير",
#     ]

#     EN_GREETINGS = [
#         "hi",
#         "hello",
#         "hey",
#         "good morning",
#         "good evening",
#         "good afternoon",
#         "hiya",
#         "yo",
#     ]

#     def is_greeting_text(msg_: str, lang_: str) -> bool:
#         """
#         Sirf tab greeting ho jab pure text me sirf greeting hi ho.
#         'chicken sandwich' jese text ko greeting na samjho.
#         """
#         if not msg_:
#             return False

#         text = msg_.strip().lower()
#         text = re.sub(r"[^\w\u0600-\u06FF ]+", "", text)

#         greetings = AR_GREETINGS if lang_ == "ar" else EN_GREETINGS
#         return text in greetings

#     # Greeting (only if stage empty)
#     if s.get("stage") is None and is_greeting_text(msg, lang):
#         s = {
#             "stage": "branch",
#             "order": [],
#             "total": 0,
#             "last_item": None,
#             "last_qty": 0,
#         }
#         session["state"] = s

#         if lang == "ar":
#             reply = (
#                 "مرحباً بك في مطعم جوانا للوجبات السريعة! 🍔<br>"
#                 "ابحث عن أقرب فرع واتصل على الرقم للطلب:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>جاهز للطلب؟ شارك طلبك بالصوت أو بالكتابة!<br><br>"
#             )
#         else:
#             reply = (
#                 "Welcome to JOANA Fast Food! <br>"
#                 "Find your nearest branch and dial the number to order:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>Ready to order? Share your order via voice or chat!<br><br>"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     if intent == "menu":
#         reply = (
#             "Here’s our menu! Please place your order."
#             if lang == "en"
#             else "هذه قائمتنا! من فضلك ضع طلبك."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang, menu="/static/menu.PNG")

#     # user says "add more" in payment (free text)
#     if s.get("stage") == "payment" and (
#         "add more" in msg.lower() or "إضافة" in msg or "اضافة" in msg or "أخرى" in msg
#     ):
#         s["stage"] = "add_more"
#         session["state"] = s
#         reply = (
#             "Sure! Please tell me what else you would like to add to your order."
#             if lang == "en"
#             else "حسناً! من فضلك أخبرني ما الذي تريد إضافته أيضاً."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return make_chat_response(reply, lang)

#     # Spicy follow-up
#     if s.get("stage") == "await_spice" and s.get("last_item"):
#         last_item = s["last_item"]
#         last_qty = s.get("last_qty", 1)
#         price, _category = get_price_and_category(last_item)
#         split = parse_spice_split(msg)

#         if split:
#             spicy_q, non_q = split
#         else:
#             spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#             if spicy_flag:
#                 spicy_q, non_q = last_qty, 0
#             elif nonspicy_flag:
#                 spicy_q, non_q = 0, last_qty
#             else:
#                 spicy_q, non_q = 0, last_qty

#         order_added_msgs = []

#         if spicy_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": spicy_q,
#                     "spicy": 1,
#                     "nonspicy": 0,
#                     "price": price,
#                     "subtotal": spicy_q * price,
#                 }
#             )
#             order_added_msgs.append(f"{spicy_q} {'spicy' if lang == 'en' else 'حار'}")

#         if non_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": non_q,
#                     "spicy": 0,
#                     "nonspicy": 1,
#                     "price": price,
#                     "subtotal": non_q * price,
#                 }
#             )
#             order_added_msgs.append(
#                 f"{non_q} {'non-spicy' if lang == 'en' else 'بدون حار'}"
#             )

#         s["stage"] = "add_more"
#         s["last_item"] = None
#         s["last_qty"] = 0
#         session["state"] = s

#         reply = (
#             f"Added {last_item.title()} — "
#             + " & ".join(order_added_msgs)
#             + ". Would you like to add anything else?"
#             if lang == "en"
#             else f"تمت إضافة {last_item} — "
#             + " و ".join(order_added_msgs)
#             + ". هل ترغب في إضافة شيء آخر؟"
#         )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # ---------- MULTI-ITEM ORDER PARSING (single message) ----------
#     segments = split_order_segments(msg)
#     if len(segments) > 1:
#         added_items_desc = []
#         for seg in segments:
#             seg_found = find_menu_item(seg)
#             if not seg_found:
#                 continue

#             qty = detect_qty(seg)
#             price, category = get_price_and_category(seg_found)

#             spicy_flag, nonspicy_flag = detect_spicy_nonspicy(seg)

#             # Burgers ke liye spicy / non-spicy handling
#             if category == "burgers_meals":
#                 if spicy_flag:
#                     spicy = 1
#                     nonspicy = 0
#                     desc_spice = "spicy" if lang == "en" else "حار"
#                 elif nonspicy_flag:
#                     spicy = 0
#                     nonspicy = 1
#                     desc_spice = "non-spicy" if lang == "en" else "بدون حار"
#                 else:
#                     # Multi-message confusion avoid karne ke liye
#                     spicy = 0
#                     nonspicy = 1
#                     desc_spice = "non-spicy" if lang == "en" else "بدون حار"
#             else:
#                 spicy = 0
#                 nonspicy = 0
#                 desc_spice = ""

#             s["order"].append(
#                 {
#                     "item": seg_found,
#                     "qty": qty,
#                     "spicy": spicy,
#                     "nonspicy": nonspicy,
#                     "price": price,
#                     "subtotal": qty * price,
#                 }
#             )

#             if lang == "ar":
#                 if desc_spice:
#                     added_items_desc.append(f"{qty} {seg_found} ({desc_spice})")
#                 else:
#                     added_items_desc.append(f"{qty} {seg_found}")
#             else:
#                 if desc_spice:
#                     added_items_desc.append(
#                         f"{qty} {desc_spice} {seg_found.title()}"
#                     )
#                 else:
#                     added_items_desc.append(f"{qty} {seg_found.title()}")

#         if added_items_desc:
#             previous_stage = s.get("stage")

#             if previous_stage == "payment":
#                 summary, total = build_order_summary_and_total(s["order"], lang)
#                 s["stage"] = "payment"
#                 s["total"] = total
#                 session["state"] = s

#                 reply = (
#                     "Your order has been updated! Here’s the new summary:<br>"
#                     + "<br>".join(summary)
#                     + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                     + "Would you like to proceed with the payment?"
#                     if lang == "en"
#                     else "تم تحديث طلبك! هذا هو الملخص الجديد:<br>"
#                     + "<br>".join(summary)
#                     + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
#                     + "هل ترغب في المتابعة إلى الدفع؟"
#                 )
#             else:
#                 s["stage"] = "add_more"
#                 session["state"] = s

#                 if lang == "en":
#                     reply = (
#                         "I’ve added the following items:<br>"
#                         + "<br>".join(added_items_desc)
#                         + "<br><br>Would you like to add anything else?"
#                     )
#                 else:
#                     reply = (
#                         "لقد قمتُ بإضافة الأصناف التالية:<br>"
#                         + "<br>".join(added_items_desc)
#                         + "<br><br>هل ترغب في إضافة شيء آخر؟"
#                     )

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)
#     # ---------- END MULTI-ITEM PARSING ----------

#     # Item detection (single-item message)
#     found = find_menu_item(msg)

#     if found:
#         qty = detect_qty(msg)
#         price, category = get_price_and_category(found)

#         s["last_item"] = found
#         s["last_qty"] = qty
#         session["state"] = s

#         spicy_flag, nonspicy_flag = detect_spicy_nonspicy(msg)

#         if category == "burgers_meals" and not (spicy_flag or nonspicy_flag):
#             s["stage"] = "await_spice"
#             session["state"] = s

#             reply = (
#                 f"Would you like your {found.title()} spicy or non-spicy?"
#                 if lang == "en"
#                 else f"هل ترغب أن يكون {found} حارًا أم بدون حار؟"
#             )

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         if category == "burgers_meals":
#             if spicy_flag:
#                 spicy = 1
#                 nonspicy = 0
#             elif nonspicy_flag:
#                 spicy = 0
#                 nonspicy = 1
#             else:
#                 spicy = 0
#                 nonspicy = 1
#         else:
#             spicy = 0
#             nonspicy = 0

#         s["order"].append(
#             {
#                 "item": found,
#                 "qty": qty,
#                 "spicy": spicy,
#                 "nonspicy": nonspicy,
#                 "price": price,
#                 "subtotal": qty * price,
#             }
#         )

#         previous_stage = s.get("stage")
#         s["last_item"] = None
#         s["last_qty"] = 0

#         if previous_stage == "payment":
#             summary, total = build_order_summary_and_total(s["order"], lang)
#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             reply = (
#                 "Your order has been updated! Here's the new summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تحديث طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )
#         else:
#             s["stage"] = "add_more"
#             session["state"] = s

#             reply = (
#                 f"{found.title()} ×{qty} added! Would you like to add anything else?"
#                 if lang == "en"
#                 else f"تمت إضافة {found} ×{qty}! هل ترغب في إضافة شيء آخر؟"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Done / no more items
#     if any(x in msg.lower() for x in ["no", "done", "that's all", "that all"]) or any(
#         x in msg for x in ["خلص", "لا", "انتهى", "كفاية"]
#     ):
#         if s.get("order"):

#             summary, total = build_order_summary_and_total(s["order"], lang)

#             reply = (
#                 "Your order is confirmed! Here's a summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تأكيد طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Payment yes
#     if s.get("stage") == "payment" and (
#         any(
#             x in msg.lower()
#             for x in [
#                 "yes",
#                 "sure",
#                 "ok",
#                 "okay",
#                 "proceed",
#                 "go ahead",
#             ]
#         )
#         or any(
#             x in msg
#             for x in [
#                 "تمام",
#                 "نعم",
#                 "أكيد",
#                 "المتابعة",
#                 "تابع",
#                 "ادفع",
#                 "إلى الدفع",
#             ]
#         )
#     ):
#         reply = (
#             "How would you like to pay?<br><b>Cash</b> or <b>Online Payment</b>?"
#             if lang == "en"
#             else "كيف ترغب في الدفع؟<br><b> نقدي </b> أو <b> الدفع عبر الإنترنت</b>؟"
#         )
#         s["stage"] = "choose_payment"
#         session["state"] = s

#         if is_voice:
#             speak_text(reply, lang)

#         return make_chat_response(reply, lang)

#     # Choose payment
#     if s.get("stage") == "choose_payment":
#         total = s.get("total", 0)
#         msg_l = msg.lower()

#         if any(x in msg_l for x in ["cash", "cod", "نقد", "نقدا"]):
#             reply = (
#                 "Thank you for your order! Your order will be ready in 20 to 30 minutes."
#                 if lang == "en"
#                 else "شكراً لطلبك! سيكون جاهزاً خلال 20 إلى 30 دقيقة."
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         elif any(
#             x in msg_l
#             for x in [
#                 "online",
#                 "online payment",
#                 "pay online",
#                 "card",
#                 "visa",
#                 "master",
#                 "debit",
#                 "mada",
#             ]
#         ) or any(x in msg for x in ["مدى", "إلكتروني", "دفع", "كرت"]):
#             reply = (
#                 f"Great! Your payment of {total:.2f} {CURRENCY} is being processed.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>Click here to complete your payment</b></a>"
#                 if lang == "en"
#                 else f"رائع! يتم الآن معالجة دفعتك بمقدار {total:.2f} {CURRENCY_AR}.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>اضغط هنا لإتمام الدفع</b></a>"
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#         else:
#             reply = (
#                 "Please select a valid payment method — Cash or Online."
#                 if lang == "en"
#                 else "يرجى اختيار طريقة الدفع الصحيحة - نقدًا أو عبر الإنترنت"
#             )
#             if is_voice:
#                 speak_text(reply, lang)

#             return make_chat_response(reply, lang)

#     # Generic LLM reply
#     reply = get_llm_reply(msg, lang)

#     if s.get("stage") == "payment" and s.get("order"):
#         summary, total = build_order_summary_and_total(s["order"], lang)
#         s["total"] = total
#         session["state"] = s

#         if lang == "en":
#             reply = (
#                 reply
#                 + "<br><br>Here is your current order summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#             )
#         else:
#             reply = (
#                 reply
#                 + "<br><br>هذا ملخص طلبك الحالي:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#     if is_voice:
#         speak_text(reply, lang)

#     return make_chat_response(reply, lang)


# # ---------------------------
# # Run App (local)
# # ---------------------------
# if __name__ == "__main__":
#     print("WHATSAPP_PHONE_NUMBER_ID =", WHATSAPP_PHONE_NUMBER_ID)
#     print("WHATSAPP_TOKEN present   =", bool(WHATSAPP_TOKEN))
#     print("WHATSAPP_VERIFY_TOKEN    =", WHATSAPP_VERIFY_TOKEN)
#     app.run(debug=True)


# import json
# import datetime
# import os
# import re
# import sqlite3
# import difflib  # (imported but not used – kept as in original)
# import requests

# import pandas as pd
# from flask import Flask, render_template, request, jsonify, session
# from nlp_utils import detect_intent, detect_language
# from openai import OpenAI

# # ---------------------------
# # ENV VARIABLES
# # ---------------------------
# from dotenv import load_dotenv

# load_dotenv()

# OPENROUTER_API_KEY = os.getenv("OPENROUTER_API_KEY", "YOUR_OPENROUTER_API_KEY_HERE")
# FLASK_SECRET = os.getenv("FLASK_SECRET_KEY", "joana-fastfood-secret")

# WATI_BASE_URL = os.getenv("WATI_BASE_URL")
# WATI_ACCESS_TOKEN = os.getenv("WATI_ACCESS_TOKEN")


# def send_whatsapp_text(to_number: str, text: str):
#     if not WATI_BASE_URL or not WATI_ACCESS_TOKEN:
#         print("WATI config missing, cannot send WhatsApp message.")
#         return

#     url = f"{WATI_BASE_URL}/api/v1/sendSessionMessage/{to_number}"

#     params = {
#         "messageText": text
#     }

#     headers = {
#         "Authorization": WATI_ACCESS_TOKEN
#     }

#     try:
#         resp = requests.post(url, headers=headers, params=params, timeout=10)
#         print("WATI send response:", resp.status_code, resp.text)
#     except Exception as e:
#         print("Error sending WhatsApp message via WATI:", e)
        
        
# def send_whatsapp_image(to_number: str, image_url: str, caption: str = ""):
#     """
#     ADVANCED version with all best practices
#     """
#     # Validation
#     if not to_number or not to_number.strip():
#         print("❌ Invalid phone number")
#         return False
        
#     if not image_url or not image_url.startswith('https://'):
#         print("❌ Invalid image URL - must be HTTPS")
#         return False

#     # URL construction
#     url = f"{WATI_BASE_URL}/api/v1/sendSessionMessage/{to_number.strip()}"
#     print(f"📤 url structure: {url}")
#     # Payload construction
#     payload = {
#         "media": {
#             "mediaType": "image",
#             "caption": caption[:1024],  # Limit caption length
#             "mediaUrl": image_url
#         }
#     }
#     print(f"📤 payload: {payload}")
#     headers = {
#         "Authorization": f"Bearer {WATI_ACCESS_TOKEN}",
#         "Content-Type": "application/json"
#     }

#     # API call with comprehensive error handling
#     try:
#         print(f"📤 Sending image to {to_number}...")
#         resp = requests.post(url, headers=headers, json=payload, timeout=59)
#         print(f"📤 response: {resp}")
#         if resp.status_code == 200:
#             print("✅ Image sent successfully!")
#             return True
#         elif resp.status_code == 401:
#             print("❌ Unauthorized - check WATI access token")
#             return False
#         elif resp.status_code == 404:
#             print("❌ Endpoint not found - check URL format")
#             return False
#         elif resp.status_code == 400:
#             print("❌ Bad request - check payload format")
#             return False
#         else:
#             print(f"❌ API Error {resp.status_code}: {resp.text}")
#             return False
            
#     except requests.exceptions.Timeout:
#         print("❌ Timeout - image may be too large or slow connection")
#         return False
#     except requests.exceptions.ConnectionError:
#         print("❌ Connection error - check internet connection")
#         return False
#     except requests.exceptions.RequestException as e:
#         print(f"❌ Request failed: {e}")
#         return False
#     except Exception as e:
#         print(f"❌ Unexpected error: {e}")
#         return False



# # ---------------------------
# # Paths / Directories
# # ---------------------------
# BASE_DIR = os.path.dirname(os.path.abspath(__file__))
# DATA_DIR = os.path.join(BASE_DIR, "data")

# # Excel files (change only these if you rename/move them)
# MENU_FILE = os.path.join(DATA_DIR, "Menu.xlsx")
# BRANCHES_FILE = os.path.join(DATA_DIR, "Branches.xlsx")

# # In-memory WhatsApp sessions: phone -> {state, messages}
# WHATSAPP_SESSIONS = {}

# # ---------------------------
# # Flask Config
# # ---------------------------
# app = Flask(__name__, static_folder="static", template_folder="templates")
# app.secret_key = FLASK_SECRET

# client = OpenAI(api_key=OPENROUTER_API_KEY, base_url="https://openrouter.ai/api/v1")

# # ---------------------------
# # Constants
# # ---------------------------
# PAYMENT_URL = (
#     "https://starlit-sopapillas-520aa2.netlify.app/"
#     "?redirect=http://127.0.0.1:5000/thankyou"
# )

# OPEN_HOUR, CLOSE_HOUR = 0, 24  # 24/7
# TAKEAWAY_ONLY = True
# CURRENCY = "SAR"
# CURRENCY_AR = "ريال سعودي"

# # ---------------------------
# # Database Setup (simple)
# # ---------------------------
# DB_PATH = os.path.join(DATA_DIR, "orders.db")  # data/orders.db (local + cPanel)


# def init_db():
#     os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
#     conn = sqlite3.connect(DB_PATH)
#     conn.execute(
#         """
#         CREATE TABLE IF NOT EXISTS orders (
#             id INTEGER PRIMARY KEY AUTOINCREMENT,
#             item TEXT,
#             qty INTEGER,
#             spicy INTEGER,
#             nonspicy INTEGER,
#             price REAL,
#             subtotal REAL,
#             total REAL,
#             payment_method TEXT,
#             datetime TEXT
#         )
#         """
#     )
#     conn.commit()
#     conn.close()


# init_db()

# # ---------------------------
# # Excel Loaders
# # ---------------------------
# def load_menu():
#     try:
#         df_raw = pd.read_excel(MENU_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             if any("name_en" in str(c).lower() for c in row):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             print("❌ Could not find header row containing 'name_en'")
#             return {}

#         df = pd.read_excel(MENU_FILE, header=header_row_index)

#         def has(col, *keys):
#             c = str(col).strip().lower()
#             return any(k in c for k in keys)

#         english_col = next((c for c in df.columns if has(c, "name_en")), None)
#         arabic_col = next((c for c in df.columns if has(c, "name_ar")), None)
#         price_col = next((c for c in df.columns if has(c, "price")), None)
#         cat_col = next(
#             (c for c in df.columns if has(c, "category", "cat", "type")), None
#         )

#         if not english_col or not price_col:
#             raise Exception("Missing name_en or price column")

#         menu = {}
#         for _, row in df.iterrows():
#             en = str(row[english_col]).strip().lower()
#             if not en or en == "nan":
#                 continue

#             ar = str(row[arabic_col]).strip().lower() if arabic_col else en
#             category = str(row.get(cat_col, "")).strip().lower() if cat_col else ""

#             try:
#                 price = float(row[price_col])
#             except Exception:
#                 continue

#             entry = {"price": price, "category": category}
#             menu[en] = entry
#             if ar:
#                 menu[ar] = entry

#         print(f"✅ Loaded {len(menu)} menu items.")
#         return menu

#     except Exception as e:
#         print("Menu load failed:", e)
#         return {}


# def load_branches():
#     try:
#         df_raw = pd.read_excel(BRANCHES_FILE, header=None)
#         header_row_index = None

#         for i, row in df_raw.iterrows():
#             row_l = [str(c).lower() for c in row]
#             if any("branch" in c for c in row_l) and any("address" in c for c in row_l):
#                 header_row_index = i
#                 break

#         if header_row_index is None:
#             header_row_index = 0

#         df = pd.read_excel(BRANCHES_FILE, header=header_row_index)

#         name_col = next((c for c in df.columns if "branch" in str(c).lower()), None)
#         addr_col = next((c for c in df.columns if "address" in str(c).lower()), None)
#         phone_col = next(
#             (
#                 c
#                 for c in df.columns
#                 if "phone" in str(c).lower() or "number" in str(c).lower()
#             ),
#             None,
#         )

#         branches = []
#         for _, row in df.iterrows():
#             branches.append(
#                 {
#                     "Branch Name": str(row.get(name_col, "")).strip(),
#                     "Address / Area": str(row.get(addr_col, "")).strip(),
#                     "Phone Number": str(row.get(phone_col, "")).strip(),
#                 }
#             )

#         print(f"✅ Loaded {len(branches)} branches.")
#         return [
#             b
#             for b in branches
#             if (b["Branch Name"] or b["Address / Area"] or b["Phone Number"])
#         ]

#     except Exception as e:
#         print("❌ Branch load failed:", e)
#         return []


# MENU = load_menu()
# BRANCHES = load_branches()


# def build_menu_context():
#     """
#     Build a text description of the whole menu from the current MENU dict.
#     This is sent to the LLM so it always sees the up-to-date menu from Excel.
#     """
#     if not MENU:
#         return "Current restaurant menu is empty."

#     lines = []
#     for name, info in MENU.items():
#         # avoid duplicating EN+AR keys: keep only names with Latin letters
#         if not re.search(r"[A-Za-z]", name):
#             continue

#         price = info.get("price", 0.0)
#         category = info.get("category", "") or ""
#         lines.append(
#             f"- {name} | price: {price:.2f} {CURRENCY} | category: {category}"
#         )

#     if not lines:
#         return "Current restaurant menu is empty."

#     return "Current restaurant menu (items, prices, categories):\n" + "\n".join(lines)


# # ---------------------------
# # Helpers
# # ---------------------------
# def html_to_whatsapp(text: str) -> str:
#     """
#     Convert Joana chatbot HTML reply into WhatsApp-friendly text.
#     - <br> -> newline
#     - <a href="tel:...">text</a> -> text  (NO URL)
#     - <a href="http...">text</a> -> text: url
#     - remove all other HTML tags
#     """
#     if not text:
#         return ""

#     # Convert <br> to newline
#     text = text.replace("<br>", "\n").replace("<br/>", "\n").replace("<br />", "\n")

#     # ----- FIXED ANCHOR HANDLER -----
#     def replace_anchor(match):
#         full_tag = match.group(0)
#         label = match.group(1).strip()

#         href_match = re.search(
#             r'href=[\'"]([^\'"]+)[\'"]',
#             full_tag,
#             flags=re.IGNORECASE,
#         )
#         if not href_match:
#             return label

#         url = href_match.group(1).strip()

#         # --- FIX: For phone numbers, DO NOT show the URL ---
#         if url.startswith("tel:"):
#             return label  # only the number

#         # For normal links, show "label: URL"
#         return f"{label}: {url}"

#     # Process all <a> tags
#     text = re.sub(
#         r"<a[^>]*>(.*?)</a>",
#         replace_anchor,
#         text,
#         flags=re.IGNORECASE | re.DOTALL,
#     )

#     # Remove remaining HTML tags
#     text = re.sub(r"<[^>]+>", "", text)

#     # Clean spaces
#     return text.strip()


# def llm_normalize_text(text: str, lang: str) -> str:
#     """
#     Use the LLM as a spelling-correction / normalization step.
#     Especially for Arabic: fix wrong spelling but keep same meaning.
#     Only used for Arabic; English is returned as-is.
#     """
#     if not text:
#         return text

#     if lang != "ar":
#         return text

#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=[
#                 {
#                     "role": "system",
#                     "content": (
#                         "You are a spell-correction engine for user messages.\n"
#                         "- The user will write in Arabic (maybe with mistakes).\n"
#                         "- Fix the spelling, normalize the text, but do NOT translate.\n"
#                         "- Do NOT add explanations, just return the corrected Arabic sentence only."
#                     ),
#                 },
#                 {"role": "user", "content": text},
#             ],
#             temperature=0,
#             max_tokens=100,
#         )
#         return res.choices[0].message.content.strip()
#     except Exception as e:
#         print("LLM normalize error:", e)
#         return text


# def detect_qty(msg: str) -> int:
#     """
#     Detect quantity from the user's message.
#     """
#     if not msg:
#         return 1

#     text = msg.lower().strip()

#     arabic_digit_map = {
#         "٠": "0",
#         "١": "1",
#         "٢": "2",
#         "٣": "3",
#         "٤": "4",
#         "٥": "5",
#         "٦": "6",
#         "٧": "7",
#         "٨": "8",
#         "٩": "9",
#     }
#     for ar, en in arabic_digit_map.items():
#         text = text.replace(ar, en)

#     digit_match = re.search(r"\b(\d+)\b", text)
#     if digit_match:
#         try:
#             q = int(digit_match.group(1))
#             if q > 0:
#                 return q
#         except ValueError:
#             pass

#     number_words_en = {
#         "one": 1,
#         "two": 2,
#         "three": 3,
#         "four": 4,
#         "five": 5,
#         "six": 6,
#         "seven": 7,
#         "eight": 8,
#         "nine": 9,
#         "ten": 10,
#     }
#     for word, value in number_words_en.items():
#         if re.search(r"\b" + re.escape(word) + r"\b", text):
#             return value

#     number_words_ar = {
#         "واحد": 1,
#         "واحدة": 1,
#         "اثنين": 2,
#         "اتنين": 2,
#         "ثنين": 2,
#         "ثلاثة": 3,
#         "ثلاث": 3,
#         "اربعة": 4,
#         "أربعة": 4,
#         "خمسة": 5,
#         "ستة": 6,
#         "سبعة": 7,
#         "ثمانية": 8,
#         "تسعة": 9,
#         "عشرة": 10,
#     }
#     for word, value in number_words_ar.items():
#         if word in text:
#             return value

#     return 1


# def polite_check(text):
#     bad = ["idiot", "stupid", "حرام", "لعنة", "غبي"]
#     return any(w in text.lower() for w in bad)


# def get_price_and_category(name):
#     entry = MENU.get(name.lower(), {})
#     return float(entry.get("price", 0)), entry.get("category", "")


# def build_order_summary_and_total(order_items, lang):
#     total = sum(i.get("subtotal", 0) for i in order_items)
#     summary = []

#     for i in order_items:
#         item_name = i["item"]
#         item_info = MENU.get(item_name.lower(), {})

#         if item_info.get("category") in ["burgers", "sandwiches"]:
#             if lang == "ar":
#                 kind = "حار" if i.get("spicy") else "بدون حار"
#             else:
#                 kind = "spicy" if i.get("spicy") else "non-spicy"
#             summary.append(f"{i['qty']} {kind} {item_name.title()}")
#         else:
#             summary.append(f"{i['qty']} {item_name.title()}")

#     return summary, total


# def speak_text(reply, lang):
#     """Backend voice disabled. Frontend handles TTS."""
#     return


# def parse_spice_split(msg):
#     text = msg.lower().replace("-", " ")
#     nums = re.findall(r"\d+", text)

#     if len(nums) >= 2:
#         n1, n2 = int(nums[0]), int(nums[1])

#         if ("spicy" in text or "حار" in text) and (
#             "non" in text or "mild" in text or "عادي" in text or "بدون" in text
#         ):
#             spicy_first = (
#                 text.find("spicy") < text.find("non") if "non" in text else True
#             )
#             arab_spicy_first = (
#                 text.find("حار") < text.find("عادي") if "عادي" in text else True
#             )

#             return (n1, n2) if (spicy_first or arab_spicy_first) else (n2, n1)

#     return None


# def find_menu_item(msg):
#     text = msg.lower()
#     candidates = []

#     for name in MENU.keys():
#         pattern = r"\b" + re.escape(name) + r"s?\b"
#         if re.search(pattern, text):
#             candidates.append(name)

#     if not candidates:
#         return None

#     candidates.sort(key=len, reverse=True)
#     return candidates[0]


# # ---------------------------
# # LLM Reply
# # ---------------------------
# def get_llm_reply(msg, lang="en"):
#     lang_name = "English" if lang == "en" else "Arabic"

#     sys_prompt = (
#         "You are Joana Fast Food Assistant.\n"
#         "Your rules:\n"
#         "- Answer ANY user question correctly — not only restaurant questions.\n"
#         "- Always keep answers SHORT, CLEAR, and TO THE POINT.\n"
#         "- Never write long paragraphs.\n"
#         "- Be very polite and friendly.\n\n"
#         "- Spelling & typos: always try to understand the user even if they type with wrong spelling, typos,\n"
#         "  missing letters, or mixed Arabic/English. Infer what they mean and answer normally, without complaining\n"
#         "  about spelling.\n\n"
#         "- If the user asks for the menu, assume the system will also show a menu image. In that case, keep your\n"
#         "  text very short, like: 'Here is our menu, please choose your items.'\n\n"
#         "Restaurant Knowledge:\n"
#         "- Joana Fast Food is open 24 hours, 7 days a week.\n"
#         "- Only takeaway service is available (no delivery).\n"
#         "- You can explain menu items, spicy preferences, and prices.\n"
#         "- You help with branches, timings, orders, and payment.\n\n"
#         "If user asks something NOT related to the restaurant:\n"
#         "- First give a SHORT and CORRECT answer.\n"
#         "- Then gently bring them back to Joana Fast Food.\n\n"
#         f"Always respond in {lang_name}.\n"
#     )

#     context = build_menu_context()

#     messages = [
#         {"role": "system", "content": sys_prompt},
#         {"role": "system", "content": context},
#     ]

#     for m in session.get("messages", []):
#         messages.append(m)

#     messages.append({"role": "user", "content": msg})

#     try:
#         res = client.chat.completions.create(
#             model="gpt-4o-mini",
#             messages=messages,
#             temperature=0.5,
#             max_tokens=250,
#         )
#         return res.choices[0].message.content.strip()

#     except Exception as e:
#         print("LLM error:", e)
#         return (
#             "Sorry, something went wrong."
#             if lang == "en"
#             else "عذراً، حدث خطأ ما."
#         )


# # ---------------------------
# # Routes
# # ---------------------------
# @app.route("/")
# def index():
#     return render_template("index.html")


# # ---------------------------
# # WATI Webhook Route (WhatsApp → Bot)
# # ---------------------------

# @app.route("/wati/webhook", methods=["POST"])
# def wati_webhook():
#     """
#     WATI se incoming WhatsApp messages receive karega.
#     """
#     data = request.get_json(force=True) or {}
#     print("Incoming from WATI:", json.dumps(data, indent=2))

#     user_text = None
#     user_number = None

#     # ---------- Payload parsing (safe) ----------
#     # Top-level waId / text
#     if "waId" in data:
#         user_number = str(data.get("waId") or "").strip()

#     if "text" in data:
#         if isinstance(data["text"], dict):
#             user_text = str(data["text"].get("body") or "").strip()
#         else:
#             user_text = str(data.get("text") or "").strip()

#     # messages[] format
#     if (not user_text) and isinstance(data.get("messages"), list) and data["messages"]:
#         msg_obj = data["messages"][0]
#         user_number = str(
#             msg_obj.get("from") or msg_obj.get("waId") or user_number or ""
#         ).strip()

#         if isinstance(msg_obj.get("text"), dict):
#             user_text = str(msg_obj["text"].get("body") or "").strip()
#         else:
#             user_text = str(msg_obj.get("text") or "").strip()

#     if not user_number or not user_text:
#         print("Could not parse incoming WATI webhook payload.")
#         return "ignored", 200

#     print(f"Processing message from {user_number}: {user_text}")

#     # ---------- REAL Joanna chatbot logic ----------
#     try:
#         result = process_whatsapp_message(user_number, user_text)

#         reply_html = result.get("reply") or "Sorry, something went wrong."
#         # HTML → plain text for WhatsApp
#         reply_text = html_to_whatsapp(reply_html)

#         # 🖼 MENU IMAGE SENDING LOGIC
#         if result.get("menu"):
#             image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#             print(f"🖼 Sending menu image to {user_number}")
            
#             # Send image with CORRECT endpoint
#             image_sent = send_whatsapp_image(
#                 user_number,
#                 image_url,
#                 caption="Here is our menu 📋"
#             )
            
#             if image_sent:
#                 print("✅ Menu image sent successfully!")
#                 # Small delay before sending text
#                 import time
#                 time.sleep(2)
#             else:
#                 print("❌ Failed to send menu image")
#                 # Send fallback text message
#                 send_whatsapp_text(user_number, "Sorry, I couldn't send the menu image. Please visit our website to view the menu.")

#         # WhatsApp par text reply (send this regardless of image)
#         send_whatsapp_text(user_number, reply_text)

#     except Exception as e:
#         print("Error in wati_webhook:", e)
#         reply_text = "Sorry, something went wrong."
#         send_whatsapp_text(user_number, reply_text)

#     return "ok", 200


# def process_whatsapp_message(user_number: str, user_text: str) -> dict:
#     """
#     WhatsApp user ke message ko existing /api/chat logic se process karta hai.
#     Yeh function:
#     - WHATSAPP_SESSIONS se us number ka state/messages nikalta hai
#     - Flask ke test_request_context se /api/chat ko POST call karta hai
#     - chat() ke baad updated session["state"] / ["messages"] wapas WHATSAPP_SESSIONS me save karta hai
#     - /api/chat ka JSON response (reply/lang/menu...) return karta hai
#     """
#     # purani state/memory nikaal lo (agar pehle se hai)
#     ctx = WHATSAPP_SESSIONS.get(user_number, {})
#     prev_state = ctx.get("state")
#     prev_messages = ctx.get("messages")

#     # Yahan hum internal request create kar rahe hain jise chat() use karega
#     with app.test_request_context(
#         "/api/chat",
#         method="POST",
#         json={
#             "message": user_text,
#             "is_voice": False,  # WhatsApp par abhi text hi treat kar rahe
#             "lang_hint": None,
#         },
#     ):
#         # is fake request ke session me purani state daal do
#         if prev_state is not None:
#             session["state"] = prev_state
#         if prev_messages is not None:
#             session["messages"] = prev_messages

#         # normal /api/chat call karein – ye tumhari sari logic run karega
#         resp = chat()  # yahi function jo @app.route("/api/chat") ke liye hai

#         # chat() ke andar jo bhi state/messages update huay, wapas store kar do
#         WHATSAPP_SESSIONS[user_number] = {
#             "state": session.get("state"),
#             "messages": session.get("messages"),
#         }

#         # Flask Response se JSON nikaal lo
#         if isinstance(resp, tuple):
#             resp_obj = resp[0]
#         else:
#             resp_obj = resp

#         data = resp_obj.get_json() or {}
#         return data


# # ---------------------------
# # Web chat route
# # ---------------------------
# @app.route("/api/chat", methods=["POST"])
# def chat():
#     global MENU, BRANCHES
#     MENU = load_menu()
#     BRANCHES = load_branches()

#     s = session.get("state", {"stage": None, "order": [], "total": 0})
#     session["messages"] = session.get("messages", [])

#     data = request.get_json(force=True)
#     msg_raw = (data.get("message") or "").strip()
#     msg = msg_raw
#     is_voice = data.get("is_voice", False)
#     lang_hint = data.get("lang_hint")

#     lang = detect_language(msg_raw)

#     if is_voice and lang_hint in ("en", "ar"):
#         lang = lang_hint

#     if lang == "ar":
#         msg = llm_normalize_text(msg_raw, lang)

#     intent = detect_intent(msg)
#     session["messages"].append({"role": "user", "content": msg})

#     # --- Force menu intent if user writes "menu" ---
#     msg_l = msg.lower()
#     if "menu" in msg_l or "send menu" in msg_l or "price list" in msg_l:
#         intent = "menu"

#     if polite_check(msg):
#         reply = "Please speak politely." if lang == "en" else "من فضلك تحدث بأدب."
#         if is_voice:
#             speak_text(reply, lang)
#         return jsonify({"reply": reply, "lang": lang})

#     AR_GREETINGS = [
#         "مرحبا",
#         "مرحبا بك",
#         "مرحبا بيك",
#         "السلام عليكم",
#         "أهلاً",
#         "أهلا",
#         "أهلاً وسهلاً",
#         "اهلا وسهلا",
#         "صباح الخير",
#         "مساء الخير",
#         "هلا",
#     ]

#     EN_GREETINGS = [
#         "hi",
#         "hello",
#         "hey",
#         "good morning",
#         "good evening",
#         "good afternoon",
#         "hiya",
#         "yo",
#     ]

#     def is_greeting_text(msg_: str, lang_: str) -> bool:
#         if not msg_:
#             return False

#         text = msg_.strip().lower()

#         if lang_ == "ar":
#             return any(g in text for g in AR_GREETINGS)

#         return any(g in text for g in EN_GREETINGS)

#     if intent == "greeting" or s["stage"] is None or is_greeting_text(msg, lang):
#         s = {
#             "stage": "branch",
#             "order": [],
#             "total": 0,
#             "last_item": None,
#             "last_qty": 0,
#         }
#         session["state"] = s

#         if lang == "ar":
#             reply = (
#                 "مرحباً بك في مطعم جوانا للوجبات السريعة! 🍔<br>"
#                 "ابحث عن أقرب فرع واتصل على الرقم للطلب:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>جاهز للطلب؟ شارك طلبك بالصوت أو بالكتابة!<br><br>"
#             )
#         else:
#             reply = (
#                 "Welcome to JOANA Fast Food! <br>"
#                 "Find your nearest branch and dial the number to order:<br><br>"
#                 + "<br>".join(
#                     [
#                         (
#                             f"<b>{b.get('Branch Name','')}</b> – "
#                             f"{b.get('Address / Area','')}<br>"
#                             f"📞 <a href='tel:{b.get('Phone Number','')}'>"
#                             f"<b>{b.get('Phone Number','')}</b></a><br>"
#                         )
#                         for b in BRANCHES
#                         if b.get("Phone Number")
#                     ][:6]
#                 )
#                 + "<br>Ready to order? Share your order via voice or chat!<br><br>"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return jsonify({"reply": reply, "lang": lang})

#     if intent == "menu":
#         reply = (
#             "Here’s our menu! Please place your order."
#             if lang == "en"
#             else "هذه قائمتنا! من فضلك ضع طلبك."
#         )
#         if is_voice:
#             speak_text(reply, lang)
#         return jsonify({"reply": reply, "menu": "/static/menu.PNG", "lang": lang})

#     if s.get("stage") == "await_spice" and s.get("last_item"):
#         last_item = s["last_item"]
#         last_qty = s.get("last_qty", 1)
#         price, _category = get_price_and_category(last_item)
#         split = parse_spice_split(msg)

#         if split:
#             spicy_q, non_q = split
#         else:
#             lower_msg = msg.lower()
#             if any(k in lower_msg for k in ["spicy", "hot", "حار"]):
#                 spicy_q, non_q = last_qty, 0
#             elif any(k in lower_msg for k in ["non", "mild", "عادي", "بدون"]):
#                 spicy_q, non_q = 0, last_qty
#             else:
#                 spicy_q, non_q = 0, last_qty

#         order_added_msgs = []

#         if spicy_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": spicy_q,
#                     "spicy": 1,
#                     "nonspicy": 0,
#                     "price": price,
#                     "subtotal": spicy_q * price,
#                 }
#             )
#             order_added_msgs.append(f"{spicy_q} {'spicy' if lang == 'en' else 'حار'}")

#         if non_q > 0:
#             s["order"].append(
#                 {
#                     "item": last_item,
#                     "qty": non_q,
#                     "spicy": 0,
#                     "nonspicy": 1,
#                     "price": price,
#                     "subtotal": non_q * price,
#                 }
#             )
#             order_added_msgs.append(
#                 f"{non_q} {'non-spicy' if lang == 'en' else 'بدون حار'}"
#             )

#         s["stage"] = "add_more"
#         s["last_item"] = None
#         s["last_qty"] = 0
#         session["state"] = s

#         reply = (
#             f"Added {last_item.title()} — "
#             + " & ".join(order_added_msgs)
#             + ". Would you like to add anything else?"
#             if lang == "en"
#             else f"تمت إضافة {last_item} — "
#             + " و ".join(order_added_msgs)
#             + ". هل ترغب في إضافة شيء آخر؟"
#         )

#         if is_voice:
#             speak_text(reply, lang)

#         return jsonify({"reply": reply, "lang": lang})

#     found = find_menu_item(msg)

#     if found:
#         qty = detect_qty(msg)
#         price, category = get_price_and_category(found)

#         s["last_item"] = found
#         s["last_qty"] = qty
#         session["state"] = s

#         incoming = msg.lower()
#         spicy_flag = any(x in incoming for x in ["spicy", "hot", "حار"])
#         nonspicy_flag = any(x in incoming for x in ["non", "mild", "عادي", "بدون"])

#         if category in ["burgers", "sandwiches"] and not (spicy_flag or nonspicy_flag):
#             s["stage"] = "await_spice"
#             session["state"] = s

#             reply = (
#                 f"Would you like your {found.title()} spicy or non-spicy?"
#                 if lang == "en"
#                 else f"هل ترغب أن يكون {found} حارًا أم بدون حار؟"
#             )

#             if is_voice:
#                 speak_text(reply, lang)

#             return jsonify({"reply": reply, "lang": lang})

#         spicy = 1 if spicy_flag else 0
#         nonspicy = 0 if spicy_flag else 1

#         s["order"].append(
#             {
#                 "item": found,
#                 "qty": qty,
#                 "spicy": spicy,
#                 "nonspicy": nonspicy,
#                 "price": price,
#                 "subtotal": qty * price,
#             }
#         )

#         previous_stage = s.get("stage")
#         s["last_item"] = None
#         s["last_qty"] = 0

#         if previous_stage == "payment":
#             summary, total = build_order_summary_and_total(s["order"], lang)
#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             reply = (
#                 "Your order has been updated! Here's the new summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تحديث طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي الجديد: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )
#         else:
#             s["stage"] = "add_more"
#             session["state"] = s

#             reply = (
#                 f"{found.title()} ×{qty} added! Would you like to add anything else?"
#                 if lang == "en"
#                 else f"تمت إضافة {found} ×{qty}! هل ترغب في إضافة شيء آخر؟"
#             )

#         if is_voice:
#             speak_text(reply, lang)

#         return jsonify({"reply": reply, "lang": lang})

#     if any(
#         x in msg.lower()
#         for x in ["no", "done", "that's all", "that all", "خلص", "لا"]
#     ):
#         if s.get("order"):

#             summary, total = build_order_summary_and_total(s["order"], lang)

#             reply = (
#                 "Your order is confirmed! Here's a summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#                 if lang == "en"
#                 else "تم تأكيد طلبك!<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#             s["stage"] = "payment"
#             s["total"] = total
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return jsonify({"reply": reply, "lang": lang})

#     if s.get("stage") == "payment" and any(
#         x in msg.lower()
#         for x in ["yes", "sure", "ok", "okay", "تمام", "نعم", "أكيد"]
#     ):
#         reply = (
#             "How would you like to pay?<br><b>Cash</b> or <b>Online Payment</b>?"
#             if lang == "en"
#             else "كيف ترغب في الدفع؟<br><b> نقدي </b> أو <b> الدفع عبر الإنترنت</b>؟"
#         )
#         s["stage"] = "choose_payment"
#         session["state"] = s

#         if is_voice:
#             speak_text(reply, lang)

#         return jsonify({"reply": reply, "lang": lang})

#     if s.get("stage") == "choose_payment":
#         total = s.get("total", 0)
#         msg_l = msg.lower()

#         if any(x in msg_l for x in ["cash", "cod", "نقد", "نقدا"]):
#             reply = (
#                 "Thank you for your order! Your order will be ready in 20 to 30 minutes."
#                 if lang == "en"
#                 else "شكراً لطلبك! سيكون جاهزاً خلال 20 إلى 30 دقيقة."
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return jsonify({"reply": reply, "lang": lang})

#         elif any(
#             x in msg_l
#             for x in [
#                 "online",
#                 "online payment",
#                 "pay online",
#                 "card",
#                 "visa",
#                 "master",
#                 "debit",
#                 "mada",
#                 "مدى",
#                 "إلكتروني",
#                 "دفع",
#                 "كرت",
#             ]
#         ):
#             reply = (
#                 f"Great! Your payment of {total:.2f} {CURRENCY} is being processed.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>Click here to complete your payment</b></a>"
#                 if lang == "en"
#                 else f"رائع! يتم الآن معالجة دفعتك بمقدار {total:.2f} {CURRENCY_AR}.<br>"
#                 f"<a href='{PAYMENT_URL}' target='_blank'>"
#                 f"<b>اضغط هنا لإتمام الدفع</b></a>"
#             )
#             s = {
#                 "stage": None,
#                 "order": [],
#                 "total": 0,
#                 "last_item": None,
#                 "last_qty": 0,
#             }
#             session["state"] = s

#             if is_voice:
#                 speak_text(reply, lang)

#             return jsonify({"reply": reply, "lang": lang})

#         else:
#             reply = (
#                 "Please select a valid payment method — Cash or Online."
#                 if lang == "en"
#                 else "يرجى اختيار طريقة الدفع الصحيحة - نقدًا أو عبر الإنترنت"
#             )
#             if is_voice:
#                 speak_text(reply, lang)

#             return jsonify({"reply": reply, "lang": lang})

#     reply = get_llm_reply(msg, lang)

#     if s.get("stage") == "payment" and s.get("order"):
#         summary, total = build_order_summary_and_total(s["order"], lang)
#         s["total"] = total
#         session["state"] = s

#         if lang == "en":
#             reply = (
#                 reply
#                 + "<br><br>Here is your current order summary:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>Total Bill: {total:.2f} {CURRENCY}</b><br>"
#                 + "Would you like to proceed with the payment?"
#             )
#         else:
#             reply = (
#                 reply
#                 + "<br><br>هذا ملخص طلبك الحالي:<br>"
#                 + "<br>".join(summary)
#                 + f"<br><br><b>الإجمالي: {total:.2f} {CURRENCY_AR}</b><br>"
#                 + "هل ترغب في المتابعة إلى الدفع؟"
#             )

#     if is_voice:
#         speak_text(reply, lang)

#     return jsonify({"reply": reply, "lang": lang})
# # ---------------------------
# # TEST IMAGE SEND FUNCTION
# # ---------------------------

# def test_image_send_manual():
#     """Test image sending with CORRECT endpoint"""
#     test_number = "923196322937"  # Use any test number
    
#     # Use your existing image URL
#     image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
    
#     print("🧪 TESTING IMAGE SEND WITH CORRECT ENDPOINT...")
#     success = send_whatsapp_image(test_number, image_url, "Test menu image")
    
#     if success:
#         print("🎉 TEST PASSED: Image sent successfully!")
#     else:
#         print("💥 TEST FAILED: Check the logs above for errors")
    
#     return success
    
# # Optional: Debug function to check WATI setup
# def debug_wati_setup():
#     """Debug your WATI configuration"""
#     print("=== WATI DEBUG INFO ===")
#     if WATI_ACCESS_TOKEN:
#         print(f"✅ WATI_ACCESS_TOKEN: {WATI_ACCESS_TOKEN[:15]}...")
#     else:
#         print("❌ WATI_ACCESS_TOKEN: NOT SET")
    
#     # Test image URL accessibility
#     image_url = "https://qintellecttechnologies.com/joana_chatbot/static/menu.PNG"
#     print(f"🔗 Image URL: {image_url}")
    
#     try:
#         response = requests.head(image_url, timeout=10)
#         if response.status_code == 200:
#             print("✅ Image URL is accessible")
#         else:
#             print(f"❌ Image URL not accessible. Status: {response.status_code}")
#     except Exception as e:
#         print(f"❌ Cannot access image URL: {e}")

# # Uncomment to test:
# # debug_wati_setup()
# # test_image_send_manual()


# # ---------------------------
# # Run App (local)
# # ---------------------------
# if __name__ == "__main__":
#     app.run(debug=True)
